/*  cardgame.c - PMODE 4 playing cards library for the CoCo.
   
    By Pierre Sarrazin <http://sarrazip.com/>.
    This file is in the public domain.
*/

#include "cardgame.h"


byte *
drawCompiledPixMap2(byte byteOffset, byte pixelRow, const word *wordArray, byte numRows,
                    byte rowRepetitions, const byte *drawingAreaEnd)
{
    assertf(byteOffset < BYTES_PER_SCREEN_ROW, "%u", byteOffset);
    assertf(pixelRow < PIXEL_ROWS_PER_SCREEN, "%u", pixelRow);
    assertf(numRows <= PIXEL_ROWS_PER_SCREEN, "%u", numRows);
    assertf(pixelRow + numRows <= PIXEL_ROWS_PER_SCREEN, "%u, %u", pixelRow, numRows);
    assertf(rowRepetitions > 0, "%u", rowRepetitions);
    
    /*  Logic implemented in assembly language that follows:
    word *dest = (word *) (scrnBuffer + (word) pixelRow * BYTES_PER_SCREEN_ROW + byteOffset);
    for (byte i = 0; i < numRows; ++i)
    {
        word w = wordArray[i];
        for (byte r = 0; r < rowRepetitions; ++r)
        {
            if (dest >= drawingAreaEnd)
                return;
            *dest = w;
            dest += WORDS_PER_SCREEN_ROW;
        }
    }
    */
    
    byte *topLeft;
    byte r;  // temp row counter
    
    asm
    {
        ldx     :scrnBuffer     // load global variable value, point X to screen start

        pshs    y               // save global data segment on OS-9

        lda     pixelRow
        ldb     #BYTES_PER_SCREEN_ROW
        mul
        addb    byteOffset
        adca    #0
        leax    d,x         // make X point to first screen byte to write to
        stx     topLeft
        
        ldy     :wordArray  // cannot refer to global vars now on OS-9

drawCompiledPixMap_foreach_word:
        ldb     rowRepetitions  // reinit rep counter
        stb     r
        ldd     ,y++

drawCompiledPixMap_foreach_rep:
        cmpx    :drawingAreaEnd         // if past the writing limit
        bhs     drawCompiledPixMap_earlyEnd

        std     ,x
        leax    BYTES_PER_SCREEN_ROW,x  // move to next row down
        
        dec     r
        bne     drawCompiledPixMap_foreach_rep
        
        dec     numRows
        bne     drawCompiledPixMap_foreach_word

drawCompiledPixMap_earlyEnd:
        puls    y
    }
    
    return topLeft;
}


byte *
drawCompiledPixMap(byte byteOffset, byte pixelRow, const word *wordArray, byte numRows, byte rowRepetitions)
{
    return drawCompiledPixMap2(byteOffset, pixelRow, wordArray, numRows,
                               rowRepetitions, scrnBuffer + 6144);  // assumes PMODE 4
}


void
drawCompiledCard2(byte cardValue, byte cardSuit, byte byteColumn, byte pixelRow,
                  char suitRowOffset, byte maxNumRowsToDraw)
{
    --cardValue;  // bring to 0..12 index

    const byte *drawingAreaEnd = scrnBuffer + (((word) pixelRow + maxNumRowsToDraw) << 5);  // 32 bytes per row

    // Draw top of card.
    drawCompiledPixMap2(byteColumn,     pixelRow, cardTopLeft,  ROWS_PER_CARD_TOP, 1, drawingAreaEnd);
    drawCompiledPixMap2(byteColumn + 2, pixelRow, cardTopRight, ROWS_PER_CARD_TOP, 1, drawingAreaEnd);

    // Draw the blank part at the right of the card value.
    drawCompiledPixMap2(byteColumn + 2, pixelRow + ROWS_PER_CARD_TOP, cardRight, 1, ROWS_PER_CARD_TOP * 2, drawingAreaEnd);

    // Draw card value (A, 2..10, J, Q, K, joker).
    const word *pixmap;
    if (cardSuit == SUIT_JOKER)
        pixmap = (cardValue == 1 ? redJokerValue : blackJokerValue);
    else
        pixmap = (word *) ((cardSuit & 1) ? blackCards[cardValue] : redCards[cardValue]);

    drawCompiledPixMap2(byteColumn, pixelRow + ROWS_PER_CARD_TOP, pixmap, ROWS_PER_CARD_VALUE, 1, drawingAreaEnd);

    // Draw blank middle (the suit will be drawn over it).
    byte middleRow = pixelRow + ROWS_PER_CARD_TOP + ROWS_PER_CARD_VALUE;
    byte middleHeight = ROWS_ABOVE_SUIT + ROWS_PER_SUIT * 2 + ROWS_BELOW_SUIT;  // times 2 b/c suit drawn w/ rowRepetitions == 2
    drawCompiledPixMap2(byteColumn,     middleRow, cardLeft,  1, middleHeight, drawingAreaEnd);
    drawCompiledPixMap2(byteColumn + 2, middleRow, cardRight, 1, middleHeight, drawingAreaEnd);
    
    // Draw bottom of card.
    drawCompiledPixMap2(byteColumn,     middleRow + middleHeight, cardBottomLeft,  ROWS_PER_CARD_BOTTOM, 1, drawingAreaEnd);
    drawCompiledPixMap2(byteColumn + 2, middleRow + middleHeight, cardBottomRight, ROWS_PER_CARD_BOTTOM, 1, drawingAreaEnd);

    // Draw suit. Use a 'rowRepetitions' of 2 to stretch the suit pixmap over 14 rows instead of 7 (ROWS_PER_SUIT).
    byte suitByteOffset = byteColumn + 1;
    byte suitRow        = pixelRow + ROWS_PER_CARD_TOP + ROWS_ABOVE_SUIT + ROWS_PER_CARD_VALUE + suitRowOffset;
    if (cardSuit == SUIT_JOKER)
        drawCompiledPixMap2(suitByteOffset, suitRow, cardValue == 1 ? redJokerSuit : blackJokerSuit, ROWS_PER_SUIT, 2, drawingAreaEnd);
    else
        drawCompiledPixMap2(suitByteOffset, suitRow, suits[cardSuit], ROWS_PER_SUIT, 2, drawingAreaEnd);
}


void
drawCompiledCard(byte cardValue, byte cardSuit, byte byteColumn, byte pixelRow)
{
    drawCompiledCard2(cardValue, cardSuit, byteColumn, pixelRow, 0, 255);
}


byte
getNumPixelRowsPerCard()
{
    return ROWS_PER_CARD_TOP + ROWS_PER_CARD_VALUE + ROWS_ABOVE_SUIT
           + ROWS_PER_SUIT * 2 + ROWS_BELOW_SUIT + ROWS_PER_CARD_BOTTOM;
}


void
eraseCard(byte byteColumn, byte pixelRow)
{
    word *p = (word *) (scrnBuffer + (((word) pixelRow) << 5) + byteColumn);
    byte h = getNumPixelRowsPerCard();
    for (byte i = 0; i < h; ++i, p += WORDS_PER_SCREEN_ROW)
    {
        p[0] = 0;
        p[1] = 0;
    }
}


void
drawFaceDownCard(byte rowInPixels, byte colInBytes)
{
    word *writer = (word *) (scrnBuffer + (word) rowInPixels * BYTES_PER_SCREEN_ROW + colInBytes);
    byte rowsPerCard = getNumPixelRowsPerCard();
    word wordToWrite = 0x5555;
    for (byte row = 0; row < rowsPerCard; ++row, writer += WORDS_PER_SCREEN_ROW)
    {
        *writer = writer[1] = wordToWrite;  // blue
        if (row == rowsPerCard - 2)
            wordToWrite = 0x5555;
        else if (row & 1)
            wordToWrite = 0x0505;
        else
            wordToWrite = 0x5050;
    }
}


byte
getCardType(byte card)
{
    if (card <= 13)
        return SUIT_HEARTS;
    if (card <= 26)
        return SUIT_SPADES;
    if (card <= 39)
        return SUIT_DIAMONDS;
    if (card <= 52)
        return SUIT_CLUBS;
    if (card <= 54)
        return SUIT_JOKER;
    return BAD_CARD_SUIT;
}


byte
getCardValue(byte card)
{
    if (card == 0)
        return BAD_CARD_VALUE;
    if (card <= 52)
        return (card - 1) % 13 + 1;
    if (card <= 54)
        return card - 52;
    return BAD_CARD_VALUE;
}


const char *
getSuitName(byte suit)
{
    switch (suit)
    {
        case SUIT_HEARTS:   return "hearts";
        case SUIT_SPADES:   return "spades";
        case SUIT_DIAMONDS: return "diamonds";
        case SUIT_CLUBS:    return "clubs";
        default:            return "ERROR";           
    }
}


void
shuffleDeck(byte deck[], byte numCards)
{
    for (byte i = 1; i < numCards; ++i)
    {
        byte other = (byte) (rand() % (numCards - i) + i);
        assertf(i - 1 < NUMCARDS, "%u", i);
        assertf(other < NUMCARDS, "%u", other);
        byte temp = deck[i - 1];
        deck[i - 1] = deck[other];
        deck[other] = temp;
    }
}


byte
pickCardFromDeck(byte deck[], byte *pNumCards)
{
    if (*pNumCards == 0)
        return BAD_CARD_VALUE;
    byte card = deck[--*pNumCards];
    assertf(card >= 1 && card <= NUMCARDS, "%u", card);
    return card;
}


void
drawSuit(byte col, byte row, byte suit)
{
    // Draw the suit by repeating each pixmap row twice:
    //
    byte *topLeft = drawCompiledPixMap(col, row, suits[suit], ROWS_PER_SUIT, 2);
    
    // Draw a white horizontal line above and below the suit pixmap just drawn at topLeft:
    //
    byte *aboveRow = topLeft - BYTES_PER_SCREEN_ROW;
    *((word *) aboveRow) = 0xFFFF;
    *((word *) (topLeft + ROWS_PER_SUIT * BYTES_PER_SCREEN_ROW * 2)) = 0xFFFF;
    
    // Draw a white vertical line at the left of that pixmap:
    //
    byte *leftSide = aboveRow - 1;
    for (byte numRows = (ROWS_PER_SUIT << 1) + 2; numRows; --numRows, leftSide += BYTES_PER_SCREEN_ROW)
        *leftSide |= 0x01;
}


void
openCardGame(void *_scrnBuffer, byte _emulatePMode4OnCoCo3)
{
    openPMode4Game(_scrnBuffer, _emulatePMode4OnCoCo3);
}


void
closeCardGame(void)
{
    closePMode4Game();
}
