/*  decbfile - A read/write DECB file library.

    By Pierre Sarrazin <http://sarrazip.com/>
    This file is in the public domain.
*/

#include "FileChoiceDialog.h"

#include "decbfile.h"

#include <assert.h>


#define min(a, b) ((a) < (b) ? (a) : (b))
#define max(a, b) ((a) > (b) ? (a) : (b))


static const char *fileDialogStrings[] =
{
//   -----------------   <- ruler that gives max length
    "FILE: ",
    "",
    "DRIVE:",
    "FILES: ?",
    "",
    "PRESS CLEAR TO",
    "CHANGE DRIVE,",
    "SLASH TO LOAD",
    "DIRECTORY, ARROWS",
    "TO SELECT FILE.",
    "ENTER=CONFIRM",
    "BREAK=CANCEL",
    "SHIFT-_=BACKSPACE"  // underscore rendered as left arrow on 6847 screen
};


static const char *fileDialogDirLoadPrompt[] =
{
    "PRESS SLASH",
    "TO LOAD THE",
    "DIRECTORY OF",
    "THE CURRENT",
    "DRIVE.",
};


// Returns the length of the longest prefix of str[0..maxLen-1]
// that does not end with paddingChar.
// maxLen: Must be > 0.
// Examples: ("FOOBAR  ", 8, ' ') gives 6.
//           ("A B C   ", 8, ' ') gives 5.
//
static byte
getUnpaddedLength(const char *str, byte maxLen, char paddingChar)
{
    for (byte i = maxLen; i--; )
        if (str[i] != paddingChar)
            return i + 1;
    return 0;
}


static void
FileNameCell_init(FileNameCell *cell, const char *name, const char *ext)
{
    byte nameLen = getUnpaddedLength(name, FileChoiceDialog_maxBasenameLen,  ' ');
    byte extLen  = getUnpaddedLength(ext,  FileChoiceDialog_maxExtLen, ' ');

    char *dest = cell->nameExt;
    memcpy(dest, name, nameLen);
    dest += nameLen;
    *dest++ = '.';
    memcpy(dest, ext, extLen);
    dest[extLen] = '\0';

    cell->next = NULL;
}


static const FileNameCell *
FileNameCell_getListElement(const FileNameCell *head, size_t index)
{
    for ( ; index > 0 && head != NULL; --index, head = head->next)
        ;
    return head;
}


// Call FileChoiceDialog_Context_close() when finished with an instance of this struct.
//
typedef struct FileChoiceDialog_Context
{
    byte *driveNo;
    byte filePathFieldCol;
    byte filePathFieldRow;
    byte dirListingCol;
    byte dirListingRow;
    byte numDirListingRows;
    byte highlightedDirListingRow;  // zero-based; index re: numDirListingRows; 255 if none
    byte driveNoFieldCol;
    byte driveNoFieldRow;
    byte numFilesFieldCol;
    byte numFilesFieldRow;
    FileNameCell *firstFileNameCell;  // linked list; max 255 elements
    byte selectedFilenameIndex;  // zero-based; index in firstFileNameCell; 255 means no selection
    const FileChoiceDialog_Terminal *term;
    FileChoiceDialog_FilterFileName filterFileName;
    FileChoiceDialog_AllocateFileNameCell allocateFileNameCell;
    FileChoiceDialog_FreeFileNameCell freeFileNameCell;
} FileChoiceDialog_Context;


// Invokes the 'freeFileNameCell' callback on each cell still in
// the 'firstFileNameCell' linked list.
// Leaves the 'firstFileNameCell' field at NULL.
//
static void
FileChoiceDialog_Context_close(FileChoiceDialog_Context *self)
{
    assert(self);
    assert(self->freeFileNameCell);
    FileNameCell *next;
    for (FileNameCell *cell = self->firstFileNameCell; cell; cell = next)
    {
        next = cell->next;
        (*self->freeFileNameCell)(cell);
    }
    self->firstFileNameCell = NULL;
}


static void
fileDialogClearDirectoryListing(FileChoiceDialog_Context *context)
{
    for (byte r = context->dirListingRow, end = r + context->numDirListingRows; r < end; ++r)
    {
        (*context->term->locate)(context->dirListingCol, r);
        (*context->term->printRepeatedChar)(FileChoiceDialog_maxBasenameLen + 1 + FileChoiceDialog_maxExtLen, ' ');
    }
    context->highlightedDirListingRow = 255;  // no highlighted row
    context->selectedFilenameIndex = 255;  // no selection
}


static void
fileDialogShowDirectoryListingPrompt(FileChoiceDialog_Context *context)
{
    for (byte i = 0; i < sizeof(fileDialogDirLoadPrompt) / sizeof(fileDialogDirLoadPrompt[0]); ++i)
        (*context->term->printAt)(context->dirListingCol, context->dirListingRow + i, fileDialogDirLoadPrompt[i]);
}


static void
fileDialogPrintDriveNo(const FileChoiceDialog_Context *context)
{
    (*context->term->locate)(context->driveNoFieldCol, context->driveNoFieldRow);
    (*context->term->printRepeatedChar)(FileChoiceDialog_maxDriveLen, ' '); // because CMOC's printf() does not support %-3u
    (*context->term->locate)(context->driveNoFieldCol, context->driveNoFieldRow);
    (*context->term->printf)("%u  ", *context->driveNo);
}


// Uses context->highlightedDirListingRow.
//
static void
FileChoiceDialog_highlightDirListingRow(const FileChoiceDialog_Context *context)
{
    if (context->highlightedDirListingRow != 255)
        (*context->term->highlightLine)(context->dirListingCol,
                           context->dirListingRow + context->highlightedDirListingRow,
                           FileChoiceDialog_maxBasenameLen + 1 + FileChoiceDialog_maxExtLen);
}


static void
FileChoiceDialog_setDirListingRowContent(const FileChoiceDialog_Context *context,
                                         const FileNameCell *newSelection)
{
    (*context->term->locate)(context->dirListingCol,
                context->dirListingRow + context->highlightedDirListingRow);
    (*context->term->printf)("%-12s", newSelection->nameExt);
}


static char
FileNameCell_compare(const FileNameCell *a, const FileNameCell *b)
{
    return (char) stricmp(a->nameExt, b->nameExt);
}


typedef char (*FileNameCell_Comparator)(const FileNameCell *, const FileNameCell *);


static BOOL
List_insertSorted(FileNameCell *head, FileNameCell *newCell, FileNameCell_Comparator compare)
{
    FileNameCell *prev = NULL;
    for ( ; head; head = head->next)
    {
        if ((*compare)(newCell, head) <= 0)  // if newCell goes before head
            break;
        prev = head;
    }

    newCell->next = head;
    if (!prev)
        return TRUE;  // newCell is new head of list
    prev->next = newCell;
    return FALSE;
}


// str: Must not be null.
// Returns FALSE if 'str' is empty.
//
static BOOL
allDecimalDigits(const char *str)
{
    if (*str == '\0')
        return FALSE;  // empty string
    while (isdigit(*str))
        ++str;
    return *str == '\0';
}


// driveNo, str: Must not be null.
//
static BOOL
convertASCIIDriveNo(byte *driveNo, const char *str)
{
    if (str[0] == '\0' || !allDecimalDigits(str))
        return FALSE;
    unsigned n = atoui(str);
    if (n > 255)
        return FALSE;
    *driveNo = (byte) n;
    return TRUE;
}


static BOOL
inputLine(char dest[], byte *inputLen, byte startCol, byte row, byte maxChars,
          const FileChoiceDialog_Terminal *terminal,
          FileChoiceDialog_Context *context);


static void
selectDrive(const FileChoiceDialog_Terminal *terminal,
            FileChoiceDialog_Context *context)
{
    char driveBuffer[FileChoiceDialog_maxDriveLen + 1];
    utoa10(*context->driveNo, driveBuffer);  // init array with current drive no
    byte inputLen;
    BOOL confirmed = inputLine(driveBuffer,
                               &inputLen, 
                               context->driveNoFieldCol,
                               context->driveNoFieldRow,
                               FileChoiceDialog_maxDriveLen,
                               terminal,
                               context);
    if (confirmed && convertASCIIDriveNo(context->driveNo, driveBuffer))
    {
        fileDialogClearDirectoryListing(context);
    }
    else  // canceled: redisplay original drive no
    {
        fileDialogPrintDriveNo(context);
    }
}


static void
printDiskErrorMessage(byte err,
                      const FileChoiceDialog_Terminal *terminal)
{
    if (err == DECB_ERR_IO)
    {
        (*terminal->printf)("I/O ERROR");
        return;
    }
    (*terminal->printf)("ERROR #%u", err);
}


static void
loadDir(const FileChoiceDialog_Terminal *terminal,
        FileChoiceDialog_Context *context)
{
    fileDialogClearDirectoryListing(context);

    DECBDirIterator iter;
    byte err = decb_openDir(*context->driveNo, &iter);
    if (err != DECB_OK)
    {
        (*context->term->locate)(context->dirListingCol, context->dirListingRow);
        printDiskErrorMessage(err, context->term);
    }
    else
    {
        FileChoiceDialog_Context_close(context);  // frees any existing list of filenames
        assert(context->firstFileNameCell == NULL);

        // Read directory entries and insert them in context->firstFileNameCell
        // so that they are in (case-insensitive) alphabetical order.
        //
        byte numFileNames = 0;
        BOOL tooManyFiles = FALSE;
        FileChoiceDialog_FilterFileName filterFileName = context->filterFileName;
        DECBDirEntry *entryPtr;

        for ( ; (err = decb_readDir(&iter, &entryPtr)) == DECB_OK; )
        {
            if (filterFileName != NULL && ! (*filterFileName)(entryPtr->name, entryPtr->ext))
                continue;

            FileNameCell *newCell = (FileNameCell *) (*context->allocateFileNameCell)();
            if (newCell == NULL)
            {
                tooManyFiles = TRUE;
                break;
            }
            FileNameCell_init(newCell, entryPtr->name, entryPtr->ext);
            if (List_insertSorted(context->firstFileNameCell, newCell, FileNameCell_compare))
                context->firstFileNameCell = newCell;
            ++numFileNames;
        }
        if (err == DECB_OK)
        {
            // Iteration ended because client's allocator failed.
            assert(tooManyFiles);
        }
        else if (err != DECB_ERR_END_OF_DIR)
        {
            // Iteration ended for reason other than end of directory, i.e., a real error.
            fileDialogClearDirectoryListing(context);
            (*context->term->locate)(context->dirListingCol, context->dirListingRow);
            printDiskErrorMessage(err, context->term);
        }
        (void) decb_closeDir(&iter);

        #if 0  // Add more elements when testing.
        for (byte i = 0; i < 15; ++i)
        {
            char name[9];
            sprintf(name, "FOOBAR%u", i);
            if (i < 10) name[7] = ' ';
            char ext[] = "XYZ";
            FileNameCell *newCell = FileNameCell_allocate(name, ext);
            if (List_insertSorted(context->firstFileNameCell, newCell, FileNameCell_compare))
                context->firstFileNameCell = newCell;
            ++numFileNames;
        }
        #endif

        // Show the number of files found in the "FILES:" field on the left side.
        // Prepend a ">" if there were more files than the client's allocator
        // was able to allocate room for.
        //
        (*context->term->locate)(context->numFilesFieldCol, context->numFilesFieldRow);
        (*context->term->printf)("%s%u  ", (tooManyFiles ? ">" : ""), numFileNames);

        // Display start of list.
        //
        const FileNameCell *it = context->firstFileNameCell;
        for (byte i = 0; i < context->numDirListingRows && it; ++i, it = it->next)
        {
            (*context->term->locate)(context->dirListingCol, context->dirListingRow + i);
            (*context->term->printf)("%-12s", it->nameExt);
        }
    }
}


// insertionOffset: Pointed byte may get modified.
//
static void
processDirListMove(char dest[], byte *inputLen, byte startCol, byte row, byte maxChars,
                   BOOL nextKeyPressed,
                   byte *insertionOffset,
                   const FileChoiceDialog_Terminal *terminal,
                   FileChoiceDialog_Context *context)
{
    const FileNameCell *newSelection = NULL;
    BOOL scrollUp = FALSE, scrollDown = FALSE;
    if (context->selectedFilenameIndex == 255)  // if no filename selected
    {
        if (context->firstFileNameCell != NULL)  // if listing has at least 1 filename
        {
            context->selectedFilenameIndex = 0;  // select 1st filename
            context->highlightedDirListingRow = 0;  // highlight 1st row of listing subwindow
            newSelection = context->firstFileNameCell;
        }
    }
    else
    {
        byte newSelectedFilenameIndex = context->selectedFilenameIndex;
        if (nextKeyPressed)
            ++newSelectedFilenameIndex;
        else
            --newSelectedFilenameIndex;  // cannot go below 0

        newSelection = FileNameCell_getListElement(context->firstFileNameCell, newSelectedFilenameIndex);
        if (newSelection != NULL)  // if new index exists in listing
        {
            context->selectedFilenameIndex = newSelectedFilenameIndex;  // confirm move
            FileChoiceDialog_highlightDirListingRow(context);  // remove highlight; uses highlightedDirListingRow

            if (nextKeyPressed)
            {
                if (context->highlightedDirListingRow < context->numDirListingRows - 1)
                    ++context->highlightedDirListingRow;
                else
                    scrollUp = TRUE;
            }
            else
            {
                if (context->highlightedDirListingRow > 0)
                    --context->highlightedDirListingRow;
                else
                    scrollDown = TRUE;
            }
        }
    } 
    if (newSelection != NULL)
    {
        asm { sync }  // avoid flicker during following screen changes

        if (scrollUp)
        {
            (*context->term->scrollSubWindowUp)(
                                    context->dirListingCol,
                                    context->dirListingRow,
                                    FileChoiceDialog_maxBasenameLen + 1 + FileChoiceDialog_maxExtLen,
                                    context->numDirListingRows);
            FileChoiceDialog_setDirListingRowContent(context, newSelection);  // uses highlightedDirListingRow
        }
        else if (scrollDown)
        {
            (*context->term->scrollSubWindowDown)(
                                    context->dirListingCol,
                                    context->dirListingRow,
                                    FileChoiceDialog_maxBasenameLen + 1 + FileChoiceDialog_maxExtLen,
                                    context->numDirListingRows);
            FileChoiceDialog_setDirListingRowContent(context, newSelection);  // uses highlightedDirListingRow
        }
        FileChoiceDialog_highlightDirListingRow(context);  // uses highlightedDirListingRow

        // Change the filename in the edit buffer.
        strcpy(dest, newSelection->nameExt);
        *insertionOffset = *inputLen = (byte) strlen(dest);
        (*context->term->clearLine)(context->filePathFieldCol, context->filePathFieldRow, FileChoiceDialog_maxPathLen);
        (*context->term->locate)(context->filePathFieldCol, context->filePathFieldRow);
        (*context->term->printf)("%s", dest);
        (*context->term->printRepeatedChar)(maxChars - *inputLen, ' ');  // clear rest of field

        // Move cursor to end of filename.
        byte cursorOffset = *insertionOffset - (*insertionOffset == maxChars);
        (*terminal->locate)(startCol + cursorOffset, row);
    }
}


// dest: In/Out. Must already contain a valid string, e.g., an empty one.
//
static BOOL
inputLine(char dest[], byte *inputLen, byte startCol, byte row, byte maxChars,
          const FileChoiceDialog_Terminal *terminal,
          FileChoiceDialog_Context *context)
{
    // Show an empty field in the targeted area.
    (*terminal->clearLine)(startCol, row, maxChars);

    // Pre-fill the field on screen with the existing contents of dest[].
    byte initLen = (byte) strlen(dest);
    if (initLen > maxChars)
        initLen = maxChars;  // ignore excess
    byte prefixLen = min(initLen, maxChars - 1);
    (*terminal->printPrefix)(dest, prefixLen);
    if (initLen == maxChars)
        (*terminal->putcharNoAdvance)(dest[prefixLen]);

    // Position the cursor at the start of the field.
    (*terminal->locate)(startCol + initLen - (initLen == maxChars), row);
    *inputLen = initLen;
    byte insertionOffset = initLen;  // offset in dest[] where next char will be inserted
    char doneKey;
    for (doneKey = 0; !doneKey; )
    {
        char key = (*terminal->waitForKeyWithAnimatedCursor)();

        // Save cursor pos: likely to be changed by loadDialogKeyHandler().
        byte origCol, origRow;
        (*terminal->getCursorPos)(&origCol, &origRow);

        switch (key)
        {
            case 3:  // Break
            case '\r':  // Enter
                dest[*inputLen] = '\0';
                doneKey = key;
                break;
            case 8:  // left arrow moves cursor left, over existing text
                if (insertionOffset > 0)  // if not a leftmost pos
                {
                    --insertionOffset;
                    (*terminal->locate)(startCol + insertionOffset, row);
                }
                break;
            case 9:  // right arrow moves cursor right, over existing text
                if (insertionOffset < *inputLen)  // if not at end of existing text
                {
                    ++insertionOffset;
                    byte cursorOffset = insertionOffset - (insertionOffset == maxChars);
                    (*terminal->locate)(startCol + cursorOffset, row);
                }
                break;
            case FileChoiceDialog_backspaceKey:
                if (insertionOffset == 0)
                    break;
                memcpy(dest + insertionOffset - 1, dest + insertionOffset, *inputLen - insertionOffset);
                // Erase last char.
                --insertionOffset;
                --*inputLen;

                // Re-print the chars that follow the one just erased.
                (*terminal->locate)(startCol + insertionOffset, row);  // positon of just-erased char
                (*terminal->printPrefix)(dest + insertionOffset, *inputLen - insertionOffset);
                (*terminal->putcharNoAdvance)(' ');

                // Put cursor over just-erased char.
                (*terminal->locate)(startCol + insertionOffset, row);
                break;

            case FileChoiceDialog_selectDriveKey:
                selectDrive(terminal, context);
                (*terminal->locate)(origCol, origRow);  // restore orig cursor pos
                break;

            case FileChoiceDialog_loadDirKey:
                loadDir(terminal, context);
                (*terminal->locate)(origCol, origRow);  // restore orig cursor pos
                break;
            
            case FileChoiceDialog_prevFilenameKey:
                if (context->selectedFilenameIndex == 0)  // if already at 1st filename
                    break;
                
                /* fallthrough */

            case FileChoiceDialog_nextFilenameKey:
                processDirListMove(dest, inputLen, startCol, row, maxChars,
                                   key == FileChoiceDialog_nextFilenameKey,
                                   &insertionOffset,
                                   terminal,
                                   context);
            break;

            default:
                if (key < ' ' || key >= 0x7F)  // if unprintable/unsupported (N.B.: codes 128..255 are -128..-1 here)
                    break;
                if (*inputLen >= maxChars)  // if no room left
                    break;
                if (insertionOffset < *inputLen)  // if there are chars at the insertion point
                    for (byte i = *inputLen; i > insertionOffset; --i)
                        dest[i] = dest[i - 1];  // move a char forward
                dest[insertionOffset] = key;
                ++*inputLen;

                // Print the char just inserted, and re-print the chars that follow it.
                byte charsToPrint = *inputLen - insertionOffset;
                if (charsToPrint > 0)
                {
                    if (*inputLen == maxChars)  // if going to print to end of screen field
                        --charsToPrint;  // don't send last char to printPrefix
                    (*terminal->printPrefix)(dest + insertionOffset, charsToPrint);
                    if (*inputLen == maxChars)
                        (*terminal->putcharNoAdvance)(dest[maxChars - 1]);  // print last char but keep cursor in field
                }

                // Advance the insertion point and put the cursor after the char just typed.
                ++insertionOffset;
                byte cursorOffset = insertionOffset - (insertionOffset == maxChars);
                (*terminal->locate)(startCol + cursorOffset, row);

        }   // switch (key)
    }
    return doneKey == '\r';
}


BOOL
FileChoiceDialog_chooseFilePath(byte *driveNo,
                                char filename[FileChoiceDialog_maxPathLen + 1],
                                const FileChoiceDialog_Terminal *terminal,
                                FileChoiceDialog_FilterFileName filterFileName,
                                FileChoiceDialog_AllocateFileNameCell allocateFileNameCell,
                                FileChoiceDialog_FreeFileNameCell freeFileNameCell)
{
    filename[0] = '\0';

    const byte col = 1, row = 1, termHeight = (*terminal->height)();

    FileChoiceDialog_Context context =
        {
            driveNo,
            col + 6, row,
            col + 18, row + 2, termHeight - 4, 255,
            col + 7, row + 2,
            col + 7, row + 3,
            NULL, 255,
            terminal,
            filterFileName,
            allocateFileNameCell,
            freeFileNameCell,
        };

    // Clear the bottom left area that shows the drive number and the instructions.
    //
    for (byte r = row + 2; r < termHeight - 1; ++r)
    {
        (*terminal->locate)(col, r);
        (*terminal->printRepeatedChar)(17, ' ');
    }
    fileDialogClearDirectoryListing(&context);

    for (byte i = 0; i < sizeof(fileDialogStrings) / sizeof(fileDialogStrings[0]); ++i)
        (*terminal->printAt)(col, row + i, fileDialogStrings[i]);

    fileDialogPrintDriveNo(&context);

    fileDialogShowDirectoryListingPrompt(&context);

    // Print a string of spaces after the "FILE:" prompt.
    //
    (*terminal->locate)(context.filePathFieldCol, context.filePathFieldRow);
    (*terminal->printRepeatedChar)(FileChoiceDialog_maxPathLen, ' ');

    for (;;)
    {
        // Prompt the user for a filename in the current drive.
        byte inputLen;
        BOOL confirmed = inputLine(filename,
                                   &inputLen,
                                   context.filePathFieldCol,
                                   context.filePathFieldRow,
                                   FileChoiceDialog_maxPathLen,
                                   terminal,
                                   &context);
        if (confirmed)  // if Enter key pressed
        {
            //term_printf("[CFP:ENTER:%s]", filename);
            //term_waitKey(TRUE);
            if (filename[0] == '\0')  // ask again if user gave empty filename
                continue;
            
            // Return a drive specification in *driveNo if any.
            char *colon = strchr(filename, ':');
            if (colon)
            {
                unsigned n = atoui(colon + 1);
                if (n > 255)
                    continue;  // invalid drive spec: ask again
                *driveNo = (byte) n;
                *colon = '\0';  // remove drive spec: caller expects filename.ext only
            }
        }

        FileChoiceDialog_Context_close(&context);  // frees any existing list of filenames

        return confirmed;
    }
}
