/*  decbfile - A read/write DECB file library.

    By Pierre Sarrazin <http://sarrazip.com/>
    This file is in the public domain.

    This file declares the FileChoiceDialog_chooseFilePath() function,
    which displays a file selection dialog in a terminal that is
    implemented by functions that are provided by the caller through
    the FileChoiceDialog_Terminal structure.
*/

#ifndef _H_FileChoiceDialog
#define _H_FileChoiceDialog

#include <coco.h>


// Element of a list of filenames.
// The library user must not assume the contents of this struct.
// It is defined here only to allow the library user to instantiate it.
// See also: FileChoiceDialog_AllocateFileNameCell.
//
typedef struct FileNameCell
{
    char nameExt[FileChoiceDialog_maxBasenameLen + 1 + FileChoiceDialog_maxExtLen + 1];
    struct FileNameCell *next;
} FileNameCell;


enum
{
    FileChoiceDialog_maxBasenameLen = 8,  // max number of chars for the pre-extension part of a filename
    FileChoiceDialog_maxExtLen = 3,  // max length of a file extension
    FileChoiceDialog_maxDriveLen = 3,  // max number of decimal digits to represent a drive no
    FileChoiceDialog_maxPathLen = FileChoiceDialog_maxBasenameLen + 1 + FileChoiceDialog_maxExtLen + 1 + FileChoiceDialog_maxDriveLen,

    // Key codes.
    FileChoiceDialog_backspaceKey = 21,  // Shift and left arrow (left arrow alone moves cursor left)
    FileChoiceDialog_selectDriveKey = 12,  // Clear key
    FileChoiceDialog_prevFilenameKey = 94,  // up arrow
    FileChoiceDialog_nextFilenameKey = 10,  // down arrow
    FileChoiceDialog_loadDirKey = '/',
};


// Function pointers that implement a text terminal.
// This allows this file choice dialog library to interface with different types of displays.
//
typedef struct FileChoiceDialog_Terminal
{
    // Moves the cursor to (startCol, row).
    // Leaves the cursor at the beginning of the cleared area.
    // numChars: Must not be zero.
    //
    void (*clearLine)(byte startCol, byte row, byte numChars);

    // Returns the number of terminal rows, which must be at least 11.
    //
    byte (*height)(void);

    // Highlights (e.g., inverts) the colors of the 'numChars' characters found at (col, row).
    // Does not move the cursor.
    //
    void (*highlightLine)(byte col, byte row, byte numChars);

    char (*waitForKeyWithAnimatedCursor)(void);

    void (*getCursorPos)(byte *column, byte *row);

    void (*locate)(byte column, byte row);

    void (*putcharNoAdvance)(int ch);

    void (*printAt)(byte col, byte row, const char *line);

    int  (*printf)(const char *fmt, ...);

    // Print the 'numChars' at 'str', without assuming that those characters
    // are followed by a NUL terminator.
    // numChars: Allowed to be zero.
    //
    void (*printPrefix)(const char *str, size_t numChars);

    void (*printRepeatedChar)(byte n, char ch);

    // Does not change the cursor location.
    //
    void (*scrollSubWindowDown)(byte leftCol, byte topRow, byte width, byte height);

    // Does not change the cursor location.
    //
    void (*scrollSubWindowUp)(byte leftCol, byte topRow, byte width, byte height);

} FileChoiceDialog_Terminal;


// Determine if the given filename must be displayed in the dialog's directory listing.
// To select all files, just return TRUE.
// name: 8-character string, space-padded, no NUL terminator.
// extension: 3-character string, space-padded, no NUL terminator.
//
typedef BOOL (*FileChoiceDialog_FilterFileName)(const char *name, const char *extension);


// Pointer type for a function that allocates a FileNameCell object
// and returns its address, or returns NULL if out of memory.
//
// N.B.: If this function returns NULL, the dialog will prepend a ">" symbol
//       to the number of files shown in the "FILES:" field, to indicate
//       that more files were found than the allocator was able to allocate
//       cells for. (For example: "FILES: >42".)
//
typedef void *(*FileChoiceDialog_AllocateFileNameCell)(void);


// Pointer type for a function that frees the given cell, allocated by
// a function of type FileChoiceDialog_AllocateFileNameCell.
//
typedef void (*FileChoiceDialog_FreeFileNameCell)(FileNameCell *cell);


// Does NOT clear the screen before showing the dialog.
// driveNo: *driveNo MUST be initialized to the drive to use initially.
//          May be changed by the user.
// filename: Does not have to be initialized.
// terminal: Text terminal functions to use to display the dialog and get user input.
//           The terminal must have at least 32 columns and 11 rows.
// filterFileName: Function that selects which files appear in the directory listing.
// allocateFileNameCell: See FileChoiceDialog_AllocateFileNameCell.
// freeFileNameCell: See FileChoiceDialog_FreeFileNameCell.
// Returns TRUE if the user wants to continue the file operation,
// FALSE to cancel it.
// If TRUE is returned, *driveNo and filename[] receive the chosen path.
//
BOOL FileChoiceDialog_chooseFilePath(byte *driveNo,
                                     char filename[FileChoiceDialog_maxPathLen + 1],
                                     const FileChoiceDialog_Terminal *terminal,
                                     FileChoiceDialog_FilterFileName filterFileName,
                                     FileChoiceDialog_AllocateFileNameCell allocateFileNameCell,
                                     FileChoiceDialog_FreeFileNameCell freeFileNameCell);


#endif  /* _H_FileChoiceDialog */
