/*  binfile.c - Support for Disk Extended Color Basic's BIN file format.

    By Pierre Sarrazin <http://sarrazip.com/>
    This file is in the public domain.
*/

#include "binfile.h"


// Private structure.
//
typedef struct
{
    bin_SectorLoadFunction sectorLoadFunction;  // must not be null
    byte *sectorBuffer;  // 256-byte buffer
    word readIndex;  // index into sectorBuffer[] (0..256); must be init to 256
    void *userData;  // passed to (*sectorLoadFunction)()
} bin_Loader;


// Private function.
//
void
bin_Loader_init(bin_Loader *loader,
                bin_SectorLoadFunction sectorLoadFunction,
                byte sectorBuffer[256],
                void *userData)
{
    loader->sectorLoadFunction = sectorLoadFunction;
    loader->sectorBuffer = sectorBuffer;
    loader->readIndex = 256;
    loader->userData = userData;
}


// Private function.
//
BOOL
bin_loadBinFileBytes(bin_Loader *loader,
                     void *dest,
                     word numBytesToRead)
{
    while (numBytesToRead > 0)
    {
        word numBytesAvailable = 256 - loader->readIndex;
        if (numBytesAvailable >= numBytesToRead)  // if enough in sectorBuffer
        {
            memcpy(dest, loader->sectorBuffer + loader->readIndex, numBytesToRead);
            loader->readIndex += numBytesToRead;
            return TRUE;
        }

        // Not enough in sectorBuffer.
        // Return what we have, then read another sector.
        memcpy(dest, loader->sectorBuffer + loader->readIndex, numBytesAvailable);
        dest += numBytesAvailable;
        numBytesToRead -= numBytesAvailable;

        if (! (*loader->sectorLoadFunction)(loader->sectorBuffer, loader->userData))
            return FALSE;  // failed to read a sector

        loader->readIndex = 0;
    }
    return TRUE;
}


BOOL
bin_loadBinFile(bin_SectorLoadFunction sectorLoadFunction,
                bin_HeaderCallback headerCallback,
                byte sectorBuffer[256],
                void *userData,
                void **entryPoint)
{
    bin_Loader loader;
    bin_Loader_init(&loader, sectorLoadFunction, sectorBuffer, userData);

    for (;;)
    {
        bin_Header header;
        if (! bin_loadBinFileBytes(&loader, &header, sizeof(header)))
            return FALSE;  // failed to read header

        if (headerCallback)
            (*headerCallback)(&header, userData);

        if (header.code == 0xFF)
        {
            if (entryPoint)
                *entryPoint = (void *) header.n1;
            return TRUE;
        }
        else if (header.code != 0)
            return FALSE;  // invalid header

        if (! bin_loadBinFileBytes(&loader, (void *) header.n1, header.n0))
            return FALSE;  // failed to read block bytes
    }
}
