/*  decbfile - A read/write DECB file library.

    By Pierre Sarrazin <http://sarrazip.com/>
    This file is in the public domain.
*/

#include "decbfile_private.h"


byte
decb_createSectorFile(DECBFile *file, byte driveNo, const char *filename,
                      byte fileType, byte fileFormat)
{
    DECBDrive *drv = decb_getDrive(driveNo);
    if (!drv)
        return DECB_ERR_DRIVE_NOT_REGISTERED;

    // Normalize the filename to 8+3 form.
    //
    char normalizedFilename[12];
    decb_normalizeFilename(normalizedFilename, filename);

    // Check if the filename is already used.
    //
    byte dirSectorBuffer[256];
    byte err = decb_findDirEntry(TRUE, driveNo, normalizedFilename, dirSectorBuffer,
                                 &file->dirSectorNum, &file->byteOffsetInDirSector);
    if (err != DECB_OK)
        return err;

    // Try to find a free granule.
    //
    file->firstGranule = decb_findFreeGranule(driveNo);  // reads the FAT
    if (file->firstGranule == 0xFF)
        return DECB_ERR_OUT_OF_SPACE;

    // Form the new dir entry.
    //
    // NOTE: The last 16 bytes of the 32-byte entry must all be zeroes,
    //       as expected by EDTASM.
    //
    DECBDirEntry *dirEntry = (DECBDirEntry *) (dirSectorBuffer + file->byteOffsetInDirSector);
    memcpy(dirEntry, normalizedFilename, 11);
    dirEntry->fileType = fileType;
    dirEntry->fileFormat = fileFormat;
    dirEntry->firstGranule = file->firstGranule;
    dirEntry->numBytesUsedInLastSector = 0;
    memset16(dirEntry->padding, 0, 8);
    err = decb_setFATEntry(driveNo, file->firstGranule, 0xC1);
    if (err != DECB_OK)
        return err;

    // Write the FAT, then the dir sector. Order is important in
    // case of failure.
    //
    err = decb_writeFAT(driveNo);
    if (err != DECB_OK)
        return err;
    if (!decb_dskcon(DECB_DSKCON_WRITE, dirSectorBuffer, driveNo, 17, file->dirSectorNum))
        return DECB_ERR_IO;
    file->driveNo = driveNo;
    file->numBytesUsedInLastSector = 0;
    file->modified = TRUE;
    return DECB_OK;
}
