/*  decbutil.h - Utility functions to use the decbfile library.

    By Pierre Sarrazin <http://sarrazip.com/>
    This file is in the public domain.

    THERE IS NO WARRANTY AS TO THE RELIABILITY OF THIS LIBRARY.
    Users as advised to make BACKUPS of any file that might come into
    contact with this library.
*/

/*  N.B.: See decbfile.h for a more flexible and lower level programming interface.
*/

#ifndef _decbutil_h_
#define _decbutil_h_

#include "decbfile.h"


// Step codes returned by decbutil_writeMemoryToFile() and decbutil_readFileToMemory().
//
enum
{
    DECBUTIL_NONE,
    DECBUTIL_START,
    DECBUTIL_DSKCON_INIT,
    DECBUTIL_INIT,
    DECBUTIL_REGISTER_DRIVE,
    DECBUTIL_SET_DSKCON_ADDRESSES,
    DECBUTIL_CREATE_SECTOR_FILE,
    DECBUTIL_OPEN_SECTOR_FILE,
    DECBUTIL_GET_FILE_SIZE,
    DECBUTIL_ALLOCATE_MEMORY,
    DECBUTIL_WRITE_SECTOR,
    DECBUTIL_READ_SECTOR,
    DECBUTIL_CLOSE_SECTOR_FILE,
    DECBUTIL_SHUTDOWN,
};


// Gives a DECB_* error code as defined by the enum in decbfile.h.
// Upon success, that code is DECB_OK.
//
#define DECBUTIL_ERR(returnedValue) ((byte) (returnedValue))


// Gives a DECBUTIL_* step code as defined by the enum above.
// Only relevant if DECBUTIL_ERR() does not give DECB_OK.
//
#define DECBUTIL_STEP(returnedValue) ((byte) ((word) (returnedValue) >> 8))


// Write a region of memory to the given drive and filename.
// Uses a standalone sector I/O routine. Does not depend on the presence of Disk Basic.
// filename: Name and extension, with a period as the extension separator (e.g., "foobar.txt").
// numSectors: Size of the memory region to write, in 256-byte sectors.
// isText: Determines if the format of the file will be text or binary.
//         DECB's DIR command will show an A or a B for the file, depending on this parameter.
// Returns a code such that DECBUTIL_ERR(code) will give a DECB_* error code and
// DECBUTIL_STEP(code) will indicate at which step the error happened.
// See the step codes enum above.
// If DECBUTIL_ERR(code) gives DECB_OK, the operation succeeded.
// The file is closed after the operation.
//
word decbutil_writeMemoryToFile(byte driveNo, const char *filename,
                                const void *memory, size_t numSectors, BOOL isText);


// Reads disk sectors from the given drive and filename into a region of memory.
// Requires at least 384 bytes of free stack space.
// numSectors: This number of sectors will be read from the start of the file.
//             The rest of the file, if any, will be ignored.
// Returns a code such that DECBUTIL_ERR(code) will give a DECB_* error code and
// DECBUTIL_STEP(code) will indicate at which step the error happened.
// See the step codes enum above.
// If DECBUTIL_ERR(code) gives DECB_OK, the operation succeeded.
// The file is closed after the operation.
//
word decbutil_readFileToMemory(byte driveNo, const char *filename,
                               void *memory, size_t numSectors);


// Like decbutil_readFileToMemory(), but accepts a callback instead of a buffer
// and buffer size.
// The callback's job is to indicate at which address the file must be loaded.
//
// If the file is found, the callback will be invoked and it will receive
// the number of sectors in the file to be loaded. The callback is allowed to
// change *numSectors to a lower number, if the client wants to load fewer
// sectors of the file.
//
// The callback also receives the size of the file in bytes, as well as a
// user data pointer, which will be userData.
//
word decbutil_readFileToDynamicMemory(byte driveNo, const char *filename,
                                      void *(*allocateMemory)(size_t *numSectors,
                                                              unsigned long fileSizeInBytes,
                                                              void *userData),
                                      void *userData);


#endif  /* _decbutil_h_ */
