/*  runDOSLoader.c - Moves and executes a loader program.

    By Pierre Sarrazin <http://sarrazip.com/>
    This file is in the public domain.

    THERE IS NO WARRANTY AS TO THE RELIABILITY OF THIS LIBRARY.
    Users as advised to make BACKUPS of any file that might come into
    contact with this library.
*/

#include "runDOSLoader.h"

#include <dskcon-standalone.h>


enum
{
    SECTORS_PER_GRANULE = 9,  // as per Disk Basic file system
    LAST_DIR_SECTOR = 18,  // as per Disk Basic file system
};


// Global structure used to pass parameters to postMoveLoaderPhase().
// Parameters cannot be pushed on the stack to postMoveLoaderPhase(),
// since the stack gets moved to high memory in the process of moving
// the loader to lower memory.
//
struct PostMoveLoaderParams
{
    char executableFilename[11];  // 11-character padded filename, e.g., "FOOBAR  APP"
    const void *clientProgramStart;  // where to load the actual application
    void *clientProgramStackBottom;  // where to reset the stack bottom just before jumping to client program
    void (*loadStartCallback)(size_t numSteps);  // called once when the client program is about to get loaded; may be null
    void (*progressCallback)(void);  // called at each "step" while the client program is getting loaded; may be null
} postMoveLoaderParams;


static BOOL
readSector(byte track, byte sector, void *dest)
{
    DCOPC = 2;  // read
    DCDRV = 0;
    DCTRK = track;
    DCSEC = sector;
    DCBPT = (byte *) dest;
    dskcon_processSector();
    return DCSTA == 0;
}


// Returns number of sectors.
//
static word
computeFileLength(const byte fat[68], byte firstGranuleNo)
{
    word numSectors = 0;
    do
    {
        firstGranuleNo = fat[firstGranuleNo];
        numSectors += (firstGranuleNo >= 0xC0 ? (firstGranuleNo & 0x0F) : SECTORS_PER_GRANULE);
    } while (firstGranuleNo < 0xC0);  // while the current granule has a successor
    return numSectors;
}


// Uses postMoveLoaderParams.
// granuleNo: First granule of the file to be loaded (0..67).
//
static BOOL
loadFile(byte granuleNo)
{
    byte fatSector[256];
    if (!readSector(17, 2, fatSector))
        return FALSE;

    word numSectors = computeFileLength(fatSector, granuleNo);

    // Print a char for each sector to load. Will serve as a progress bar.
    if (postMoveLoaderParams.loadStartCallback)
        (*postMoveLoaderParams.loadStartCallback)(numSectors);

    for (byte *dest = (byte *) postMoveLoaderParams.clientProgramStart; ; )
    {
        byte g = fatSector[granuleNo];
        byte numSectorsToLoad = (g >= 0xC0 ? (g & 0x0F) : SECTORS_PER_GRANULE);

        // Tracks 0..16 contain granules 0..33.
        // Tracks 18- contain granules 34-.
        byte track = (granuleNo >> 1) + (granuleNo <= 33 ? 0 : 1);
        byte sec = ((granuleNo & 1) ? 10 : 1);  // even granules in sectors 1..9, odd ones in 10..18
        for (byte i = 1; i <= numSectorsToLoad; ++i, ++sec, dest += 256)
        {
            if (!readSector(track, sec, dest))
                return FALSE;
            if (postMoveLoaderParams.progressCallback)
                (*postMoveLoaderParams.progressCallback)();
        }
 
        if (g >= 0xC0)  // if last granule
            break;
        
        granuleNo = g;  // loop to read next granule
    }

    return TRUE;
}


// Must be called upon IRQ during disk operations.
//
static interrupt asm void
irqISR(void)
{
    asm
    {
        tst     $FF03
        bpl     @done
        ; 60 Hz interrupt.
        ldb     $FF02       ; Reset PIA0, port B interrupt flag.
        lbsr    dskcon_irqService
@done
    }
}


// Function that gets called to continue the loading phase once the loader
// has been moved to lower memory.
// Interrupts are assumed to be masked upon entry to this function.
// Uses postMoveLoaderParams.
// Installs irqISR() as the IRQ handler.
// Installs dskcon_nmiService() as the NMI handler.
//
static byte
postMoveLoaderPhase(void)
{
    asm  // Point IRQ to new addres of irqISR(), while interrupts are masked.
    {
        leax    irqISR          ; PCR mode used here, so this loads the post-move address of irqISR()
        tfr     x,d
        ldx     $FFF8           ; IRQ vector
        std     1,x             ; store address argument of JMP 
        ldb     #$7E            ; JMP
        stb     ,x
    }
    dskcon_init(dskcon_nmiService);
    enableInterrupts();

    // Scan dir sectors and load postMoveLoaderParams.executableFilename.
    byte dirSector[256];
    for (byte sec = 3; sec <= LAST_DIR_SECTOR; ++sec)
    {
        if (!readSector(17, sec, dirSector))
            return 1;  // program not found
        for (byte i = 0; i < 8; ++i)  // for each entry of the sector
        {
            const byte *entry = dirSector + ((word) i << 5);
            if (entry[0] == 0xFF)  // if end of dir
            {
                sec = LAST_DIR_SECTOR;  // stop scanning
                break;
            }
            if (entry[0] != 0x00 && memcmp(entry, postMoveLoaderParams.executableFilename, 11) == 0)
            {
                if (!loadFile(entry[13]))  // pass file's 1st granule no
                    return 2;  // failed to load client program

                disableInterrupts();
                asm
                {
                    ; Reset stack to reuse space used by this function.
                    lds     :postMoveLoaderParams.clientProgramStackBottom

                    jmp     [:postMoveLoaderParams.clientProgramStart]
                }
            }
        }
    }
    return 1;  // program not found
}


byte
decbfile_runDOSLoader(void *loaderDestination,
                        const void *loaderSource,
                        void *newStackBottom,
                        const char *executableFilename,
                        void *clientProgramStart,
                        void (*loadStartCallback)(size_t numSteps),
                        void (*progressCallback)(void))
{
    // Pass parameters to postMoveLoaderPhase() (before moving the code and globals
    // to lower/higher memory).
    memcpy(postMoveLoaderParams.executableFilename, executableFilename, 11);
    postMoveLoaderParams.clientProgramStart = clientProgramStart;
    postMoveLoaderParams.clientProgramStackBottom = newStackBottom;

    // Compute the new address of postMoveLoaderPhase().
    word delta = (word) loaderDestination - (word) loaderSource;
    byte (*postMoveFunction)(void) = postMoveLoaderPhase + delta;

    // Compute and pass the new addresses of the callback functions.
    postMoveLoaderParams.loadStartCallback = loadStartCallback + delta;
    postMoveLoaderParams.progressCallback  = progressCallback + delta;

    // Mask the interrupts now because the following memcpy() may put code
    // in a region where Color Basic's IRQ routine may write (e.g., $0985,
    // which is Disk Basic's RDYTMR variable, i.e., the motor turn-off timer).
    disableInterrupts();

    // Copy track 34 from (typically) the address where the DOS command loaded it
    // to loaderDestination.
    // N.B.: postMoveLoaderParams must have been fully filled at this point.
    memcpy(loaderDestination, loaderSource, 18 * 256);

    byte errorCode;
    asm
    {
        ldx     :postMoveFunction   ; new address of postMoveLoaderPhase(); this instruction relies on U as stack frame ptr

        clr     $FFDF               ; put CoCo in all RAM mode (already the case on a CoCo 3)

        lds     :newStackBottom     ; stack space now at end of RAM; this instruction also relies on U

        jsr     ,x                  ; execute postMoveLoaderPhase() at its new address, with interrupts still masked

        stb     :errorCode          ; JSR returns upon error, so store error code [relies on U to reach 'errorCode']
    }

    return errorCode;
}
