// Program that tests decbfie.h
//
// Must be run from drive 0, with an empty disk in drive 1.

#include "decbfile_private.h"
#include "crc16.h"


#define showError(err) do { if (err != DECB_OK) printf("\n""ERROR #%u at line %d\n", err, __LINE__); } while (0)


// N.B.: Evaluates its arguments twice.
#define min(a, b) ((a) < (b) ? (a) : (b))


enum { numSectorsInFile = 23 };


byte killIfPresent(byte driveNo, const char *filename)
{
    byte err = decb_kill(driveNo, filename);
    if (err == DECB_ERR_NOT_FOUND)
        return DECB_OK;
    return err;
}


byte testFileCreation(const char *filename, byte driveNo, uint16_t perSectorCRC[numSectorsInFile])
{
    printf("TEST: testFileCreation\n");
    DECBFile file;
    byte err = decb_createSectorFile(&file, driveNo, filename,
                                DECB_TYPE_BASIC_DATA, DECB_FORMAT_ASCII);

    if (err != DECB_OK)
        return err;

    byte dataSectorBuffer[256];
    for (word i = 0; i < 256; i += 8)
    {
        char tmp[9];
        sprintf(tmp, "$%06x\r", i);
        memcpy(dataSectorBuffer + i, tmp, 8);
    }

    for (byte fileSectorIndex = numSectorsInFile; fileSectorIndex--; )
    {
        sprintf((char *) dataSectorBuffer, "FILE SECTOR %02u.\r",
                                            fileSectorIndex);
        perSectorCRC[fileSectorIndex] = crc16_block(
                                                CRC16_RESET,
                                                dataSectorBuffer,
                                                sizeof(dataSectorBuffer));
        err = decb_writeSector(&file, dataSectorBuffer, fileSectorIndex);

        printf("%3u %%   ", (numSectorsInFile - fileSectorIndex) * (word) 100 / numSectorsInFile);
    }
    printf("\n");

    decb_setNumBytesUsedInLastSector(&file, 16);
    err = decb_closeSectorFile(&file);

    printf("PER-SECTOR CRC16 VALUES:\n");
    for (byte fileSectorIndex = 0; fileSectorIndex < numSectorsInFile; ++fileSectorIndex)
    {
        printf("%2u=$%04x  ", fileSectorIndex, perSectorCRC[fileSectorIndex]);
    }
    printf("\n");

    return DECB_OK;
}


byte testFileRead(const char *filename, byte driveNo, uint16_t perSectorCRC[numSectorsInFile])
{
    printf("TEST: testFileRead\n");
    DECBFile file;
    memset(&file, 0, sizeof(file));  // for test
    byte err = decb_openSectorFile(&file, driveNo, filename);
    if (err != DECB_OK)
    {
        return err;
    }

    printf("fileType=%u, fileFormat=%u, lastSector: %u\n",
           file.fileType, file.fileFormat, file.numBytesUsedInLastSector);
    byte dataSectorBuffer[256];

    for (byte fileSectorIndex = 0; fileSectorIndex < numSectorsInFile; ++fileSectorIndex)
    {
        //printf("- READING SECTOR %u OF FILE\n", fileSectorIndex);
        memset(dataSectorBuffer, '$', 256);  // for test
        err = decb_readSector(&file, dataSectorBuffer, fileSectorIndex);
        if (err != DECB_OK)
        {
            printf("ERROR: decb_readSector failed\n");
            exit(1);
        }

        uint16_t crc = crc16_block(CRC16_RESET,
                                   dataSectorBuffer,
                                   sizeof(dataSectorBuffer));
        if (crc != perSectorCRC[fileSectorIndex])
        {
            printf("ERROR: got CRC $%04x, expected $%04x\n",
                            crc, perSectorCRC[fileSectorIndex]);
            printf("Press key to see read data: ");
            waitkey(TRUE);
            printf("\n");
            putstr((char *) dataSectorBuffer, 256);
            printf("\n");
            exit(1);
        }
        printf("%3u %%   ", (fileSectorIndex + 1) * (word) 100 / numSectorsInFile);
    }
    printf("\n");

    err = decb_closeSectorFile(&file);
    if (err != DECB_OK)
        printf("ERROR: decb_closeSectorFile failed\n");

    return DECB_OK;
}


byte copyProgressFunctor(word currentSectorIndex,
                         word totalNumSectors,
                         void *userData)
{
    if (isKeyPressed(0xFF - (1 << 2), 1 << 6))  // test Break key
        return FALSE;

    return TRUE;  // continue
}


// Copies all files from drive 0 to drive 1, then kills all files in
// drive 1.
//
byte testFileCopy()
{
    printf("TEST: testFileCopy\n");
    printf("Press BREAK to interrupt this test.\n");

    DECBDirIterator dirIter;
    byte err = decb_openDir(0, &dirIter);
    if (err != DECB_OK)
    {
        printf("ERROR: decb_openDir() failed on copy\n");
        return err;
    }
    for (;;)
    {
        DECBDirEntry *dirEntry;
        err = decb_readDir(&dirIter, &dirEntry);
        if (err == DECB_ERR_END_OF_DIR)
        {
            err = DECB_OK;
            break;  // normal end of dir
        }
        if (err != DECB_OK)
        {
            printf("\nERROR: decb_readDir() failed on copy: error #%u\n", err);
            break;
        }

        char fn[13];
        decb_denormalizeFilename(fn, dirEntry->name, FALSE);
        printf("Copying %s ", fn);

        err = decb_copyFile(0, dirEntry, 1, fn, copyProgressFunctor, 0);
        if (err == DECB_ERR_INTERRUPTED)
        {
            printf("\nCopy interrupted.\n");
            break;
        }
        if (err != DECB_OK)
        {
            printf("\nERROR: decb_copyFile() failed: error #%u\n", err);
            break;
        }
        printf("\n");
    }

    byte closeErr = decb_closeDir(&dirIter);
    if (err != DECB_OK)
        return err;
    if (closeErr != DECB_OK)
        return closeErr;

    err = decb_openDir(1, &dirIter);
    if (err != DECB_OK)
    {
        printf("ERROR: decb_openDir() failed on kill\n");
        return err;
    }
    for (;;)
    {
        DECBDirEntry *dirEntry;
        err = decb_readDir(&dirIter, &dirEntry);
        if (err == DECB_ERR_END_OF_DIR)
        {
            err = DECB_OK;
            break;  // normal end of dir
        }
        if (err != DECB_OK)
        {
            printf("\nERROR: decb_readDir() failed on copy: error #%u\n", err);
            break;
        }

        char fn[13];
        decb_denormalizeFilename(fn, dirEntry->name, FALSE);
        printf("Killing %s\n", fn);

        err = decb_kill(1, fn);
        if (err != DECB_OK)
        {
            printf("\nERROR: decb_kill() failed: error #%u\n", err);
            break;
        }
    }

    closeErr = decb_closeDir(&dirIter);
    if (err != DECB_OK)
        return err;
    return closeErr;
}


typedef struct ConcatTestData
{
    DECBFile destFile;
    DECBSeqWriteBuffer writeBuffer;
    dword concatLengthInBytes;
    word sourceCRC16;
    word destCRC16;
} ConcatTestData;


byte flushConcatData(ConcatTestData *data)
{
    if (!decb_hasFullSector(&data->writeBuffer))  // if not enough to write a sector
        return DECB_OK;

    // Compute the running CRC-16 of the source files.
    // Must be done before decb_flush(), which removes the 256 bytes from the buffer.
    //
    data->destCRC16 = crc16_block(data->destCRC16, data->writeBuffer.buffer, 256);

    byte err = decb_flush(&data->destFile, &data->writeBuffer);
    if (err != DECB_OK)
    {
        printf("ERROR: Failed to write sector to destination file: error #%u\n", err);
        return err;
    }

    return DECB_OK;
}


byte concatFile(byte driveNo, DECBDirEntry *dirEntry, ConcatTestData *data)
{
    char fn[13];
    decb_denormalizeFilename(fn, dirEntry->name, FALSE);

    // Get number of sectors and bytes in source file.
    word numSectors;
    dword fileLen;
    (void) decb_getFileSizeFromDirEntry(driveNo, dirEntry, &numSectors, &fileLen);

    // Print source filename and length.
    printf("%-13s  %6lu\n", fn, fileLen);

    // Open source file.
    DECBFile srcFile;
    byte err = decb_openSectorFileFromDirEntry(&srcFile, driveNo, dirEntry);
    if (err != DECB_OK)
    {
        printf("ERROR: Failed to open %s: error #%u\n", fn, err);
        return FALSE;  // stop iterating through dir
    }

    // Copy each source file sector.
    //
    for (word fileSectorIndex = 0; fileSectorIndex < numSectors; ++fileSectorIndex)
    {
        // Point to available part of 'data' buffer.
        byte *sectorBuffer = decb_getFreeSpaceAddress(&data->writeBuffer);

        // Read one sector.
        err = decb_readSector(&srcFile, sectorBuffer, fileSectorIndex);
        if (err != DECB_OK)
        {
            printf("ERROR: Failed to read sector #%u from %s: error #%u\n",
                                                    fileSectorIndex, fn, err);
            break;
        }

        // Determine number of file bytes read.
        word numBytesRead;
        if (fileSectorIndex + 1 == numSectors)  // if last sector of file
            numBytesRead = dirEntry->numBytesUsedInLastSector;
        else
            numBytesRead = 256;

        // Compute the running CRC-16 of the source files.
        data->sourceCRC16 = crc16_block(data->sourceCRC16, sectorBuffer, numBytesRead);

        // Update the data buffer and flush it (if it contains a complete sector).
        decb_registerWrittenBytes(&data->writeBuffer, numBytesRead);
        err = flushConcatData(data);

        if (err != DECB_OK)
        {
            printf("ERROR: Failed to write to destination file: error #%u\n", err);
            break;
        }
    }

    // Close source file.
    byte closeErr = decb_closeSectorFile(&srcFile);
    if (err != DECB_OK)
    {
        printf("Quitting test.\n");
        return FALSE;
    }
    if (closeErr != DECB_OK)
    {
        printf("ERROR: Failed to close %s: error #%u\n", fn, err);
        return FALSE;
    }

    // Sum source file lengths.
    data->concatLengthInBytes += fileLen;

    return TRUE;  // continue to next file
}


// Also tests truncation.
//
byte testFileConcatenation()
{
    printf("TEST: testFileConcatenation\n");

    // Init test data.
    ConcatTestData data;
    data.concatLengthInBytes = 0;
    data.sourceCRC16 = CRC16_RESET;
    data.destCRC16   = CRC16_RESET;
    decb_initSeqWriteBuffer(&data.writeBuffer);

    // Erase data file if present.
    byte destDriveNo = 1;
    const char *destFilename = "CONCAT.DAT";
    byte err = killIfPresent(destDriveNo, destFilename);
    if (err != DECB_OK)
    {
        printf("ERROR: Failed to erase %s: error #%u\n", destFilename, err);
        return FALSE;
    }

    // Create data file.
    err = decb_createSectorFile(&data.destFile, destDriveNo, destFilename,
                                DECB_TYPE_BASIC_DATA, DECB_FORMAT_BINARY);
    if (err != DECB_OK)
    {
        printf("ERROR: decb_createSectorFile() failed: error #%u\n", err);
        return FALSE;
    }

    // For each file in the directory, append its contents to the data file.
    DECBDirIterator dirIter;
    err = decb_openDir(0, &dirIter);
    if (err != DECB_OK)
    {
        printf("ERROR: decb_openDir() failed\n");
        return err;
    }

    for (;;)
    {
        DECBDirEntry *dirEntry;
        err = decb_readDir(&dirIter, &dirEntry);
        if (err == DECB_ERR_END_OF_DIR)
        {
            err = DECB_OK;
            break;  // normal end of dir
        }
        if (err != DECB_OK)
        {
            printf("\nERROR: decb_readDir() failed: error #%u\n", err);
            break;
        }

        if (!concatFile(0, dirEntry, &data))
            break;
    }

    // Close the dir. iterator.
    byte closeErr = decb_closeDir(&dirIter);
    if (err != DECB_OK)
        return err;
    if (closeErr != DECB_OK)
    {
        printf("ERROR: Failed to directory: error #%u\n", err);
        return closeErr;
    }


    // Remember the number of bytes to be written to the file's last sector.
    // The numUsedBytes field is about to be overwritten by decb_flush().
    //
    word numBytesInBuffer = data.writeBuffer.numUsedBytes;

    if (err != DECB_OK)
    {
        printf("ERROR: decb_listDirectory() failed: error #%u\n", err);
    }
    else if (numBytesInBuffer != 0)
    {
        // The 'data' buffer is not empty, so one last sector receives the remaining bytes.
        // Erase the rest of the sector buffer, to avoid leaking data from one file to another.
        //
        memset(decb_getFreeSpaceAddress(&data.writeBuffer), 0xFF, 256 - numBytesInBuffer);

        data.destCRC16 = crc16_block(data.destCRC16, data.writeBuffer.buffer, numBytesInBuffer);

        err = decb_flush(&data.destFile, &data.writeBuffer);
    }

    // Close the destination file and tell it the size of its last sector.
    //
    word numBytesLastSector = (numBytesInBuffer == 0 ? (word) 256 : numBytesInBuffer);
    decb_setNumBytesUsedInLastSector(&data.destFile, numBytesLastSector);
    closeErr = decb_closeSectorFile(&data.destFile);

    if (closeErr != DECB_OK)
    {
        printf("ERROR: Failed to close %s: error #%u\n", destFilename, closeErr);
        return FALSE;
    }

    if (err != DECB_OK)
        return FALSE;

    printf("Total file lengths: %lu bytes\n", data.concatLengthInBytes);
    printf("Source CRC     : $%04x\n", data.sourceCRC16);
    printf("Destination CRC: $%04x\n", data.destCRC16);

    // Truncate destination file as a test.
    {
        word numSectors[]            = { 907,610,610,  72, 52, 52, 48,  18, 16, 16,   9,  9,  7,  7,  3,  1,  1, 1 };
        word numBytesInLastSector[]  = {  42, 42, 42, 256, 42, 42, 42, 256, 42, 42, 256, 42, 42, 42, 42, 42, 42, 0 };
        byte numTests = sizeof(numSectors) / sizeof(numSectors[0]);

        for (byte i = 0; i < numTests; ++i)
        {
            printf("Truncation at %u sectors, %u bytes in last sector\n",
                                    numSectors[i], numBytesInLastSector[i]);

            DECBFile bigFile;
            err = decb_openSectorFile(&bigFile, destDriveNo, destFilename);
            if (err != DECB_OK)
            {
                printf("ERROR: Failed to open %s: error #%u\n", destFilename, err);
                return FALSE;
            }

            err = decb_truncateOpenFile(&bigFile, numSectors[i], numBytesInLastSector[i]);
            printf(" decb_truncateOpenFileAtSectorOffset() returned %u\n", err);

            if (err == DECB_ERR_INVALID_ARGUMENT && numSectors[i] > DECB_MAX_NUM_GRANULES * 9)
                continue;  // expected error

            if (err != DECB_OK)
            {
                printf("ERROR: decb_truncateOpenFileAtSectorOffset() failed: error #%u\n", err);
                return FALSE;
            }

            // Check the new size reported by decb_getNumGranulesInOpenFile().
            byte lastFileGranule = 0;
            word sectorsInFile = 0;
            byte numGranules = decb_getNumGranulesInOpenFile(&bigFile, &lastFileGranule, &sectorsInFile);
            if (numGranules == 0xFF)
            {
                printf("ERROR: decb_getNumGranulesInOpenFile() failed\n");
                return FALSE;
            }

            byte checkFileLength = (numSectors[i] < 610);
            if (checkFileLength)
            {
                byte expectedNumGrans = (byte) ((numSectors[i] + 8) / 9);
                if (expectedNumGrans == 0)
                    expectedNumGrans = 1;
                if (numGranules != expectedNumGrans)
                {
                    printf("ERROR: got %u granules, expected %u\n", numGranules, expectedNumGrans);
                    return FALSE;
                }
                if (sectorsInFile != numSectors[i])
                {
                    printf("ERROR: got %u sectors, expected %u\n", sectorsInFile, numSectors[i]);
                    return FALSE;
                }
            }

            err = decb_closeSectorFile(&bigFile);
            if (err != DECB_OK)
            {
                printf("ERROR: decb_closeSectorFile failed: error #%u\n", err);
                return FALSE;
            }


            char normalizedFilename[12];
            decb_normalizeFilename(normalizedFilename, destFilename);
            byte dirSectorBuffer[256];
            byte dirSectorNum = 0, byteOffsetInDirSector = 0;
            err = decb_findDirEntry(FALSE, destDriveNo, normalizedFilename,
                    dirSectorBuffer, &dirSectorNum, &byteOffsetInDirSector);
            if (err != DECB_OK)
            {
                printf("ERROR: decb_findDirEntry(): error #%u\n", err);
                return FALSE;
            }
            DECBDirEntry *dirEntry = (DECBDirEntry *) (dirSectorBuffer + byteOffsetInDirSector);
            printf(" Dir entry: 1st gran: %u; bytes in last sector: %u\n",
                    dirEntry->firstGranule, dirEntry->numBytesUsedInLastSector);

            if (checkFileLength)
            {
                dword lengthInBytes = 0;
                sectorsInFile = 0;
                numGranules = decb_getFileSizeFromDirEntry(destDriveNo, dirEntry, &sectorsInFile, &lengthInBytes);
                if (numGranules == 0xFF)
                {
                    printf("ERROR: decb_getFileSizeFromDirEntry() failed\n");
                    return FALSE;
                }
                printf(" Length in bytes: %lu. In sectors: %u. In granules: %u\n",
                        lengthInBytes, sectorsInFile, numGranules);
                if (sectorsInFile != numSectors[i])
                {
                    printf("ERROR: decb_getFileSizeFromDirEntry(): got %u sectors, expected %u\n",
                            sectorsInFile, numSectors[i]);
                    return FALSE;
                }
                if ((numBytesInLastSector[i] & 0xFF) == 0)  // if 0 or 256
                {
                    if ((lengthInBytes & 0x00FF) != 0)
                    {
                        printf("ERROR: decb_getFileSizeFromDirEntry(): length in bytes not divisible by 256 as expected\n");
                        return FALSE;
                    }
                }
                else if ((lengthInBytes & 0x00FF) != numBytesInLastSector[i])  // 1..255
                {
                    printf("ERROR: decb_getFileSizeFromDirEntry(): got %lu bytes in last sector, expected %u\n",
                            lengthInBytes & 0x00FF, numBytesInLastSector[i]);
                    return FALSE;
                }

                // Test decb_getFileSizeFromFilename() by comparing its
                // results with those from decb_getFileSizeFromDirEntry().
                //
                word altNumSectors = 0;
                dword altLengthInBytes = 0;
                byte altNumGranules = decb_getFileSizeFromFilename(
                                            destDriveNo, destFilename, &altNumSectors, &altLengthInBytes);
                if (altNumGranules != numGranules
                        || altNumSectors != sectorsInFile
                        || altLengthInBytes != lengthInBytes)
                {
                    printf("ERROR: decb_getFileSizeFromFilename: got %u, %u, %lu instead of %u, %u, %lu\n",
                            altNumGranules, altNumSectors, altLengthInBytes,
                            numGranules, sectorsInFile, lengthInBytes);
                    return FALSE;
                }
            }
        }
    }


    // Rename destination file as a test.
    //
    const char *newDestFilename = "concat.new";
    printf("Renaming %s to %s\n", destFilename, newDestFilename);
    err = decb_rename(destDriveNo, destFilename, newDestFilename);
    if (err != DECB_OK)
    {
        printf("ERROR: Failed to rename %s to %s: error #%u\n",
                destFilename, newDestFilename, err);
        return FALSE;
    }


    // Erase destination file to clean up.
    //
    printf("Cleaning up %s\n", newDestFilename);
    err = decb_kill(destDriveNo, newDestFilename);
    if (err != DECB_OK)
    {
        printf("ERROR: Failed to clean up %s: error #%u\n", newDestFilename, err);
        return FALSE;
    }

    if (data.destCRC16 != data.sourceCRC16)
        return FALSE;

    return TRUE;
}


byte checkRecordFileLength(DECBRecordFile *recFile, word expectedNumRecords)
{
    if (recFile->numRecords != expectedNumRecords)
    {
        printf("ERROR: recFile->numRecords=%u, expected %u\n", recFile->numRecords, expectedNumRecords);
        return FALSE;
    }
    return TRUE;
}


// Assumes that a floppy is in Drive 1.
//
byte testRecordFileTruncation(word headerSize)
{
    printf("TEST: testRecordFileTruncation(%u)\n", headerSize);

    byte destDriveNo = 1;
    const char *destFilename = "RECORDS.DAT";
    if (killIfPresent(destDriveNo, destFilename) != DECB_OK)
    {
        printf("ERROR: Failed to erase %s:%u\n", destFilename, destDriveNo);
        return FALSE;
    }

    word recordSize = 113;
    byte dummyRecord[113];

    DECBRecordFile recFile;
    byte err = decb_createRecordFile(&recFile, destDriveNo, destFilename, headerSize, recordSize);
    showError(err);
    if (err != DECB_OK)
        return FALSE;
    if (recFile.headerSize != headerSize)
    {
        printf("ERROR: recFile.headerSize=%u, expected %u\n", recFile.headerSize, headerSize);
        return FALSE;
    }
    if (recFile.recordSize != recordSize)
    {
        printf("ERROR: recFile.recordSize=%u, expected %u\n", recFile.recordSize, recordSize);
        return FALSE;
    }

    word initNumRecords = 70;
    err = decb_writeRecord(&recFile, dummyRecord, initNumRecords - 1);
    showError(err);
    if (err != DECB_OK)
        return FALSE;

    word recordIndexToTruncateAt = 96;
    for (;;)
    {
        if (!checkRecordFileLength(&recFile, min(recordIndexToTruncateAt, initNumRecords)))
            return FALSE;

        if (recordIndexToTruncateAt >= 13)
            recordIndexToTruncateAt -= 13;
        else if (recordIndexToTruncateAt > 0)
            recordIndexToTruncateAt = 0;
        else
            break;

        printf("  Truncating file at %u records.\n", recordIndexToTruncateAt);
        err = decb_truncateOpenRecordFile(&recFile, recordIndexToTruncateAt);
        showError(err);
        if (err != DECB_OK)
            return FALSE;

        // Check file length.
        word numSectors;
        dword lengthInBytes;
        byte grans = decb_getOpenFileSize(&recFile.file, &numSectors, &lengthInBytes);
        dword expectedLengthInBytes = min(recordIndexToTruncateAt, initNumRecords)
                                         * (dword) recordSize
                                           + headerSize;
        printf("    Got length %lu\n", lengthInBytes);
        if (grans == 0xFF || lengthInBytes != expectedLengthInBytes)
        {
            printf("ERROR: %u granules, expected %lu\n", grans, expectedLengthInBytes);
            return FALSE;
        }
    }

    err = decb_closeRecordFile(&recFile);
    showError(err);
    if (err != DECB_OK)
        return FALSE;

    if (killIfPresent(destDriveNo, destFilename) != DECB_OK)
    {
        printf("ERROR: Failed to erase %s:%u\n", destFilename, destDriveNo);
        return FALSE;
    }

    return TRUE;
}


typedef struct DirListingStats
{
    word numFiles;
} DirListingStats;


byte yesToAll = FALSE;


byte processDirEntry(byte driveNo, DECBDirEntry *dirEntry, DirListingStats *stats)
{
    ++stats->numFiles;
    printf("%2u. ", stats->numFiles - 1);
    putstr(dirEntry->name, 8);
    putchar('.');
    putstr(dirEntry->ext, 3);
    printf(" %u %c ",
            dirEntry->fileType,
            dirEntry->fileFormat == DECB_FORMAT_ASCII ? 'A' : 'B');
    word numSectors;
    dword lengthInBytes;
    byte numGrans = decb_getFileSizeFromDirEntry(driveNo, dirEntry, &numSectors, &lengthInBytes);
    if (numGrans == 0xFF)
        printf("??? ???? ??????");
    else
    {
        printf("%3u %4u %6lu", numGrans, numSectors, lengthInBytes);
    }
    putchar('\n');

    if (!yesToAll && stats->numFiles % 23 == 0)
    {
        printf("--More--");
        byte key = (byte) toupper(waitkey(TRUE));
        putchar('\n');
        if (key == 'N')
            return FALSE;
    }

    return TRUE;
}


byte testDirListing()
{
    printf("TEST: testDirListing\n");

    DirListingStats stats;
    stats.numFiles = 0;

    DECBDirIterator dirIter;
    byte err = decb_openDir(0, &dirIter);
    if (err != DECB_OK)
    {
        printf("ERROR: decb_openDir() failed\n");
        return err;
    }

    for (;;)
    {
        DECBDirEntry *dirEntry;
        err = decb_readDir(&dirIter, &dirEntry);
        if (err == DECB_ERR_END_OF_DIR)
        {
            err = DECB_OK;
            break;  // normal end of dir
        }
        if (err != DECB_OK)
        {
            printf("\nERROR: decb_readDir() failed: error #%u\n", err);
            break;
        }

        if (!processDirEntry(0, dirEntry, &stats))
            break;
    }

    printf("Listed %u file(s).\n", stats.numFiles);

    byte closeErr = decb_closeDir(&dirIter);
    if (err != DECB_OK)
        return err;
    return closeErr;
}


byte checkRecordFileContents(DECBRecordFile *recFile, byte *singleRecordBuffer, word expectedNumRecords)
{
    word actualNumRecords = decb_getNumRecords(recFile);
    if (actualNumRecords != expectedNumRecords)
    {
        printf("ERROR: File contains %u record(s), expected %u.\n", actualNumRecords, expectedNumRecords);
        return FALSE;
    }

    // Check header.
    byte err = decb_readHeader(recFile, singleRecordBuffer);
    showError(err);
    if (err != DECB_OK)
        return FALSE;
    byte expectedHeaderByte = ~ (byte) (recFile->headerSize & 0xFF);
    for (word i = 0; i < recFile->headerSize; ++i)
        if (singleRecordBuffer[i] != expectedHeaderByte)
        {
            printf("ERROR: Header byte at offset %u is $%02x, expected $%02x.\n",
                    i, singleRecordBuffer[i], expectedHeaderByte);
            return FALSE;
        }

    // Check each record.
    for (word r = 0; r < expectedNumRecords + 1; ++r)
    {
        printf("  Checking record %u\n", r);
        byte err = decb_readRecord(recFile, singleRecordBuffer, r);
        if (r >= expectedNumRecords)
        {
            if (err != DECB_ERR_NOT_FOUND)
            {
                printf("ERROR: got code %u, expected %u\n", err, DECB_ERR_NOT_FOUND);
                return FALSE;
            }
            continue;
        }
        showError(err);
        if (err != DECB_OK)
            return FALSE;

        byte b = (byte) r;
        for (word i = 0; i < recFile->recordSize; ++i)
        {
            if (singleRecordBuffer[i] != b)
            {
                printf("ERROR: Record #%u has error at record offset %u: got $%02x, expected $%02x.\n",
                        r, i, singleRecordBuffer[i], b);
                return FALSE;
            }
            b += (byte) r;
        }
    }

    return TRUE;
}


// Read/write test of DECBRecordFile.
// singleRecordBuffer: Must be large enough for a header also.
//
byte testRecordFile(word headerSize, word recordSize, byte *singleRecordBuffer)
{
    printf("TEST: testRecordFile (headerSize=%u, recordSize=%u)\n", headerSize, recordSize);

    byte destDriveNo = 1;
    const char *destFilename = "RECORDS.DAT";
    byte err = killIfPresent(destDriveNo, destFilename);
    if (err != DECB_OK)
    {
        printf("ERROR: Failed to erase %s:%u: error #%u\n",
               destFilename, destDriveNo, err);
        return FALSE;
    }

    DECBRecordFile recFile;
    err = decb_createRecordFile(&recFile, destDriveNo, destFilename, headerSize, recordSize);
    showError(err);
    if (err != DECB_OK)
        return FALSE;

    // Compose and write a header.
    for (word i = 0; i < headerSize; ++i)
        singleRecordBuffer[i] = ~ (byte) (headerSize & 0xFF);
    err = decb_writeHeader(&recFile, singleRecordBuffer);
    showError(err);
    if (err != DECB_OK)
        return FALSE;

    word recordIndexes[] = { 3, 1, 4, 2, 0 };
    word numRecords = sizeof(recordIndexes) / sizeof(recordIndexes[0]);

    for (word r = 0; r < numRecords; ++r)
    {
        word recordIndex = recordIndexes[r];
        printf("  Writing record #%u\n", recordIndex);

        // Fill the record with a pattern that depends on the record's index.
        byte b = (byte) recordIndex, prev = 0;
        for (word i = 0; i < recordSize; ++i)
        {
            singleRecordBuffer[i] = b;
            b += (byte) recordIndex;
        }

        err = decb_writeRecord(&recFile, singleRecordBuffer, recordIndex);
        showError(err);
        if (err != DECB_OK)
            return FALSE;
    }

    if (!checkRecordFileContents(&recFile, singleRecordBuffer, numRecords))
        return FALSE;

    err = decb_closeRecordFile(&recFile);
    showError(err);
    if (err != DECB_OK)
        return FALSE;

    printf("  Reopening and re-checking the file.\n");

    err = decb_openRecordFile(&recFile, destDriveNo, destFilename, headerSize, recordSize);
    showError(err);
    if (err != DECB_OK)
        return FALSE;

    if (!checkRecordFileContents(&recFile, singleRecordBuffer, numRecords))
        return FALSE;

    err = decb_closeRecordFile(&recFile);
    showError(err);
    if (err != DECB_OK)
        return FALSE;

    if (killIfPresent(destDriveNo, destFilename) != DECB_OK)
    {
        printf("ERROR: Failed to erase %s:%u\n", destFilename, destDriveNo);
        return FALSE;
    }

    return TRUE;
}


byte promptForNextTest()
{
    printf("Next test (Yes/No/All)? ");
    char key = (yesToAll ? 'Y' : (char) toupper(waitkey(TRUE)));
    printf("%c\n", key);
    if (key == 'A')
    {
        yesToAll = TRUE;
        return TRUE;
    }
    return key == 'Y';
}


#ifdef _CMOC_CHECK_STACK_OVERFLOW_
void stackOverflowHandler(char *addressOfFailedCheck, char *stackRegister)
{
    printf("[FAIL: %p, %p]\n", addressOfFailedCheck, stackRegister);
    exit(1);
}
#endif


asm void *getStackPointer()
{
    asm
    {
        tfr     s,d
        addd    #2
    }
}


asm void getSBrkLimits(void **_program_end, void **_program_break, void **_end_of_sbrk_mem, void **inistk)
{
    asm
    {
program_break   IMPORT
end_of_sbrk_mem IMPORT
INISTK          IMPORT
        leax    program_end,pcr
        stx     [2,s]
        ldd     program_break,pcr
        std     [4,s]
        ldd     end_of_sbrk_mem,pcr
        std     [6,s]
        ldd     INISTK,pcr
        std     [8,s]
    }
}


int main()
{
#ifdef _CMOC_CHECK_STACK_OVERFLOW_
    void *initS;
    asm { sts initS };
    printf("Init S: %p\n", initS);
    set_stack_overflow_handler(stackOverflowHandler);
#endif

    initCoCoSupport();
    setHighSpeed(FALSE);  // typically preferable for disk access

    if (isCoCo3)
    {
        resetPalette(TRUE);
        width(80);
        cls(3);  // clean screen with blue background
        attr(3, 2, FALSE, FALSE);  // white text on blue background
    }
    else
        cls(255);

    printf("THIS PROGRAM REQUIRES AN EMPTY FLOPPY IN DRIVE 1.\nPRESS A KEY TO CONTINUE.\n");
    waitkey(TRUE);

    DECBDrive drives[2];  // only two drives are used by this program
    if (decb_init(drives, 2) != 0)
    {
        printf("ERROR: initialization failed\n");
        return 1;
    }

    // Register the two drive numbers to be used (0 and 1).
    for (byte driveNo = 0; driveNo <= 1; ++driveNo)
        if (decb_registerDrive(driveNo) != DECB_OK)
        {
            printf("ERROR: failed to register drive %u\n", driveNo);
            return 1;
        }

    const char *filename = "NEW.DAT";  // could be lowercase
    byte driveNo = 0;

    void *program_end, *program_break, *end_of_sbrk_mem, *inistk;
    getSBrkLimits(&program_end, &program_break, &end_of_sbrk_mem, &inistk);
    printf("program_end=%p; program_break=%p; end_of_sbrk_mem=%p; inistk=%p\n",
                                program_end, program_break, end_of_sbrk_mem, inistk);
    printf("sbrkmax: %u; stack pointer: %p\n", sbrkmax(), getStackPointer());

    do
    {
        byte *singleRecord = (byte *) sbrk(2540);
        printf("singleRecord: %p\n", singleRecord);
        if (singleRecord == -1)
        {
            printf("ERROR: sbrk() failed\n");
            return 1;
        }

        word headerSizes[] = { 0, 113, 2540 };
        word recordSizes[] = { 71, 499, 999, 2500 };

        byte h;
        for (h = 0; h < sizeof(headerSizes) / sizeof(headerSizes[0]); ++h)
        {
            byte r;
            for (r = 0; r < sizeof(recordSizes) / sizeof(recordSizes[0]); ++r)
            {
                if (!testRecordFile(headerSizes[h], recordSizes[r], singleRecord))
                    break;

                if (!promptForNextTest())
                    break;
            }
            if (r < sizeof(recordSizes) / sizeof(recordSizes[0]))
                break;
        }
        if (h < sizeof(headerSizes) / sizeof(headerSizes[0]))
            break;

        if (!testRecordFileTruncation(0))
            break;

        if (!promptForNextTest())
            break;

        if (!testRecordFileTruncation(300))
            break;

        if (!promptForNextTest())
            break;

        if (!testFileConcatenation())
            break;

        if (!promptForNextTest())
            break;

        if (testFileCopy() != DECB_OK)
            break;

        if (!promptForNextTest())
            break;

        if (testDirListing() != DECB_OK)
            break;

        if (!promptForNextTest())
            break;

        byte err = killIfPresent(driveNo, filename);
        if (err != DECB_OK)
        {
            printf("ERROR: failed to kill %s:%u: code #%u\n", filename, driveNo, err);
            break;
        }

        uint16_t perSectorCRC[numSectorsInFile];

        if (testFileCreation(filename, driveNo, perSectorCRC) != DECB_OK)
            break;

        if (!promptForNextTest())
            break;

        if (testFileRead(filename, driveNo, perSectorCRC) != DECB_OK)
            break;
    } while (FALSE);

    if (decb_unregisterDrive(0) != DECB_OK)
    {
        printf("ERROR: failed to unregister drive 0\n");
        return 1;
    }

    if (decb_unregisterAllDrives() != DECB_OK)
    {
        printf("ERROR: failed to unregister remaining drives\n");
        return 1;
    }

    if (decb_shutdown() != 0)
    {
        printf("ERROR: shutdown failed\n");
        return 1;
    }

    printf("Normal end.\n");
    return 0;
}
