/*  decbfile - A read/write DECB file library.

    By Pierre Sarrazin <http://sarrazip.com/>
    This file is in the public domain.
*/

#include "decbfile_private.h"


byte
decb_writeSector(DECBFile *file,
                 const byte sectorBuffer[256],
                 word fileSectorIndex)
{
    DECBDrive *drv = decb_getDrive(file->driveNo);
    if (!drv)
        return DECB_ERR_DRIVE_NOT_REGISTERED;  // more accurate than DECB_ERR_CORRUPT_FAT

    // Get the length of the file in granules.
    // Also get the FAT entry index of the file's last granule.
    //
    byte lastFileGranule;
    word numSectors;  // not used
    byte numGranulesInFile = decb_getNumGranulesInOpenFile(file, &lastFileGranule, &numSectors);
    if (numGranulesInFile == 0xFF)
        return DECB_ERR_CORRUPT_FAT;

    // fileSectorIndex could be beyond the end of the file.
    // Allocate granules and sectors to the file until it is not.
    //
    byte fileGranIndex = (byte) (fileSectorIndex / 9);
    while (fileGranIndex >= numGranulesInFile)
    {
        byte allocatedGranule = decb_findFreeGranule(file->driveNo);
        if (allocatedGranule >= DECB_MAX_NUM_GRANULES)
            return DECB_ERR_OUT_OF_SPACE;
        byte err = decb_setFATEntry(file->driveNo, lastFileGranule, allocatedGranule);
        if (err != DECB_OK)
            return err;
        err = decb_setFATEntry(file->driveNo, allocatedGranule, 0xC1);
        if (err != DECB_OK)
            return err;
        lastFileGranule = allocatedGranule;
        ++numGranulesInFile;
    }

    // Here, fileGranIndex is now inside the file's currently allocated granules.
    // Now check that this granule has enough sectors for fileSectorIndex.
    // If not, extend this granule.
    //
    byte g;  // the granule to write to
    byte err = decb_getFileGranuleFromIndex(file, fileGranIndex, &g);
    if (err != DECB_OK)
        return err;
    byte nextGran = drv->fatBuffer[g];
    if (!decb_isValidFATEntry(nextGran))
        return DECB_ERR_CORRUPT_FAT;
    byte numSectorsUsedInGranule = (nextGran < 0xC1 ? 9 : nextGran & 0x0F);
    byte sectorIndexInGranule = (byte) (fileSectorIndex % 9);
    if (sectorIndexInGranule >= numSectorsUsedInGranule)  // if index not valid
    {
        numSectorsUsedInGranule = sectorIndexInGranule + 1;  // make it valid
        err = decb_setFATEntry(file->driveNo, g, 0xC0 | numSectorsUsedInGranule);  
        if (err != DECB_OK)
            return err;
    }

    // Here, fileSectorIndex is now inside the file's currently allocated sectors.
    // Write the data sector to the disk.
    byte track, sector;
    err = decb_computeTrackAndSector(file, fileGranIndex, sectorIndexInGranule,
                                     &track, &sector);
    if (err != DECB_OK)
        return err;
    if (!decb_dskcon(DECB_DSKCON_WRITE, (void *) sectorBuffer, file->driveNo, track, sector))
        return DECB_ERR_IO;
    file->modified = TRUE;
    return DECB_OK;
}
