/*  decbdrivewire.h

    By Pierre Sarrazin <http://sarrazip.com/>
    This file is in the public domain.

    THERE IS NO WARRANTY AS TO THE RELIABILITY OF THIS LIBRARY.
    Users as advised to make BACKUPS of any file that might come into
    contact with this library.
*/

#include "decbdrivewire.h"


enum
{
    // From the DriveWire protocol.
    OP_READEX = 0xD2,
    OP_WRITE = 0x57,
};


// Pointers to functions that read and write byte sequences.
DWReadFuncPtr dwRead;
DWWriteFuncPtr dwWrite;


static BOOL
sendPacket(byte operationCode, byte driveNo, unsigned long lsn)
{
    // Send a packet that requests the sector.
    byte packet[5];
    packet[0] = operationCode;
    * (unsigned long *) (packet + 1) = lsn;  // write lsn to packet[1..4]
    packet[1] = driveNo;
    /*printf("#SP: %02x %02x %02x %02x %02x\n",
            packet[0], packet[1], packet[2], packet[3], packet[4]);*/
    return dwWrite(packet, sizeof(packet)) == 0;
}


static word
dwCrc16(const byte src[], word numBytes)
{
    word sum = 0;
    for (word i = 0; i < numBytes; ++i)
        sum += src[i];
    return sum;
}


static word
sendChecksumWaitErrorCode(word cksum)
{
    // Send a checksum.
    if (dwWrite(&cksum, sizeof(cksum)) != 0)
        return 0x102;  // failed to send checksum

    // Wait for an error code.
    byte serverErrorCode;
    byte err = dwRead(&serverErrorCode, sizeof(serverErrorCode));
    if (err != 0)
        return 0x103;  // failed to receive server error code

    return serverErrorCode;
}


static word
readDriveWireSector(byte driveNo, unsigned long lsn, byte dest[256])
{
    if (!sendPacket(OP_READEX, driveNo, lsn))
        return 0x100;  // failed to send request

    // Wait for the sector contents, and compute its checksum.
    byte err = dwRead(dest, 256);
    //printf("#RDWS: err=$%02x\n", err);
    if (err != 0)
        return 0x104;  // framing error, or not all bytes received

    word cksum = dwCrc16(dest, 256);
    //printf("#RDWS: cksum=$%04x\n", cksum);
    return sendChecksumWaitErrorCode(cksum);
}


static word
writeDriveWireSector(byte driveNo, unsigned long lsn, const byte src[256])
{
    if (!sendPacket(OP_WRITE, driveNo, lsn))
        return 0x100;  // failed to send request

    // Send the sector contents.
    if (dwWrite(src, 256) != 0)
        return 0x101;  // failed to send sector
    
    word cksum = dwCrc16(src, 256);
    return sendChecksumWaitErrorCode(cksum);
}


static DECBDskConVariables dwDskConVars;


// Reads dwDskConVars, updates its status field.
//
void
static dwDskCon()
{
    asm
    {
        pshs    u,y,x,b,a   ; preserve registers for safety
    }

    // A cast to word is necessary under CMOC, otherwise the multiplication gives a byte.
    // 1 gets subtracted from the sector field, which is 1..18, because the computation of
    // a DriveWire logical sector number requires counting a track's sector from 0.
    //
    word lsn = dwDskConVars.track * (word) DECB_LAST_DIR_SECTOR + (dwDskConVars.sector - 1);
    word result;

    /*printf("#DSKCON: O=%u D=%u T=%u S=%u (LSN=%u) B=%p\n",
                dwDskConVars.opCode,
                dwDskConVars.drive, dwDskConVars.track, dwDskConVars.sector, lsn,
                dwDskConVars.buffer);*/
    if (dwDskConVars.opCode == DECB_DSKCON_READ)
    {
        result = readDriveWireSector(dwDskConVars.drive, lsn, (byte *) dwDskConVars.buffer);
    }
    else if (dwDskConVars.opCode == DECB_DSKCON_WRITE)
    {
        result = writeDriveWireSector(dwDskConVars.drive, lsn, (byte *) dwDskConVars.buffer);
    }
    //printf("#DSKCON: RESULT=$%x\n", result);

    // Return "drive not ready" upon error (to be improved?).
    dwDskConVars.status = (result == 0 ? 0 : 0x80);

    asm
    {
        puls    a,b,x,y,u
    }
}


void
decb_initDriveWireSupport(DWReadFuncPtr dwReadFuncPtr,
                          DWWriteFuncPtr dwWriteFuncPtr)
{
    dwRead = dwReadFuncPtr;
    dwWrite = dwWriteFuncPtr;
    decb_setDskConAddresses(dwDskCon, &dwDskConVars);
}
