/*  dwdemo.c - Demo of using DriveWire via the Becker port.

    By Pierre Sarrazin <http://sarrazip.com/>
    This file is in the public domain.

    THERE IS NO WARRANTY AS TO THE RELIABILITY OF THIS LIBRARY.
    Users as advised to make BACKUPS of any file that might come into
    contact with this library.
*/

/*
    In this demo, the communication with the DriveWire server
    does not require the presence of Disk Basic.

    To run this demo:
    -   Build dwdemo.dsk by building the decbfile library (with make).
    -   Launch a DriveWire server that mounts dwdemo.dsk as Drive 0. 
    -   Launch a CoCo emulator that runs HDB-DOS (e.g., with hdbdw3bck.rom).
    -   Load the demo (via DriveWire) with LOADM "DWDEMO".
    -   Run it with EXEC.
*/

#include <decbdrivewire.h>
#include <dskcon-standalone.h>


///////////////////////////////////////////////////////////////////////////////////////////////////


int
main()
{
    cls(255);
    printf("DRIVEWIRE DEMO\n\n");

    disableInterrupts();
    unsigned long initReturnValue = dskcon_init(dskcon_nmiService);
    enableInterrupts();

    DECBDrive driveArray[1];  // only use one disk drive
    byte err = decb_init(driveArray, sizeof(driveArray) / sizeof(driveArray[0]));
    if (err != 0)
    {
        printf("FAILED INIT\n");
        return 1;
    }
    decb_registerDrive(0);  // set the disk drive to be drive #0

    // Initialize DriveWire support by passing it the addresses of
    // functions that read and write byte sequences from and to
    // the DriveWire server.
    // Here, we use functions that use the Becker port.
    //
    printf("INITIALIZING DRIVEWIRE SUPPORT.\n");
    decb_initDriveWireSupport(dwReadBeckerPort, dwWriteBeckerPort);

    printf("OPENING DIRECTORY IN DRIVE 0\n""VIA DRIVEWIRE...\n");
    DECBDirIterator dirIter;
    err = decb_openDir(0, &dirIter);
    if (err != 0)
    {
        printf("FAILED OPENDIR\n");
    }
    else
    {
        printf("PRESS A KEY TO LIST THE FILES\n"
               "IN THE DIRECTORY... ");
        waitkey(TRUE);
        printf("\n"
               "THIS STEP WILL FREEZE IF NO\n"
               "DRIVEWIRE SERVER IS RUNNING.\n");
        word numEntriesPrinted = 0, numLinesPrinted = 0;
        for (;;)
        {
            DECBDirEntry *dirEntry;
            err = decb_readDir(&dirIter, &dirEntry);
            if (err != DECB_OK)
                break;
            printf("%2u ", numEntriesPrinted);
            putstr(dirEntry->name, 8);
            putchar('.');
            putstr(dirEntry->ext, 3);
            ++numEntriesPrinted;
            if (numEntriesPrinted & 1)
                printf(" ");
            else
            {
                printf("\n");
                ++numLinesPrinted;
                if (numLinesPrinted > 0 && numLinesPrinted % 15 == 0)
                {
                    printf("--MORE--");
                    waitkey(TRUE);
                    printf("\n");
                }
            }
        }
        if (numEntriesPrinted & 1)
            printf("\n");
        if (err != DECB_ERR_END_OF_DIR)
            printf("FAILED READDIR: ERROR #%u\n", err);
        else
            printf("%u ENTRIES LISTED.\n", numEntriesPrinted);
        err = decb_closeDir(&dirIter);
    }

    // Read and show a text file.
    DECBFile file;
    byte demoTextDriveNo = 0;
    const char *demoTextFilename = "DEMOTEXT.TXT";
    printf("OPENING %s:%u...\n", demoTextFilename, demoTextDriveNo);
    err = decb_openSectorFile(&file, demoTextDriveNo, demoTextFilename);
    if (err != DECB_OK)
    {
        printf("FAILED TO OPEN %s:%u\n", demoTextFilename, demoTextDriveNo);
    }
    else
    {
        word numSectors;
        dword lengthInBytes;
        byte numGranules = decb_getOpenFileSize(&file, &numSectors, &lengthInBytes);
        if (numGranules == 255)
        {
            printf("FAILED TO GET SIZE OF %s:%u\n", demoTextFilename, demoTextDriveNo);
        }
        else
        {
            printf("FILE HAS %lu BYTES.\n", lengthInBytes);
            byte sectorBuffer[256];
            err = decb_readSector(&file, sectorBuffer, 0);  // assuming file fits in single sector
            if (err != DECB_OK)
            {
                printf("FAILED TO READ SECTOR FROM %s:%u\n", demoTextFilename, demoTextDriveNo);
            }
            else
            {
                printf("FILE CONTENTS:\n-------------------------------\n");
                putstr((char *) sectorBuffer, (size_t) lengthInBytes);
                printf("-------------------------------\n");
            }
        }
        err = decb_closeSectorFile(&file);
        if (err != DECB_OK)
        {
            printf("FAILED TO CLOSE %s:%u\n", demoTextFilename, demoTextDriveNo);
        }
    }

    decb_shutdown();
    dskcon_shutdown(initReturnValue);

    printf("NORMAL END.\n");
    return 0;
}
