/*  fcdemo.c - Demo for the FileChoiceDialog object on the 32x16 (6847) screen.

    By Pierre Sarrazin <http://sarrazip.com/>
    This file is in the public domain.

    THERE IS NO WARRANTY AS TO THE RELIABILITY OF THIS LIBRARY.
    Users as advised to make BACKUPS of any file that might come into
    contact with this library.
*/

#include "FileChoiceDialog.h"

#include "decbfile.h"
//#include <dskcon-standalone.h>  /* for dskcon_nmiService(); provided by CMOC */


enum
{
    VIDRAM = 0x0400,
    MAX_FILES_PER_DISK = 68,
    MAIN_SCREEN_BG_COLOR = 3,  // cls() argument
};


#if 1  /* SG4 (32x16) terminal. */


byte
termHeight(void)
{
    return 16;
}


byte *
getLocationAddress(byte column, byte row)
{
    return (byte *) VIDRAM + ((word) row << 5) + column;  // specific to 32x16 screen
}


// True lowercase must NOT be enabled (on the CoCo that support it).
//
void
highlightLine(byte col, byte row, byte numChars)
{
    byte *dest = getLocationAddress(col, row);
    for (byte i = 0; i < numChars; ++i)
        dest[i] ^= 0x40;
}


BOOL
getCursorAnimationStep(void)
{
    return ((byte) getTimer() & 0x38) == 0x38;
}


// True lowercase must NOT be enabled (on the CoCo that support it).
//
char
waitKeyWithAnimatedCursor(void)
{
    char *cursorPos = * (char **) 0x0088;  // Basic's cursor position
    BOOL colorsInverted = FALSE;
    setTimer(0);
    BOOL prevAnimStep = getCursorAnimationStep();
    for (;;)
    {
        *cursorPos ^= 0x40;  // invert colors
        colorsInverted = !colorsInverted;

        BOOL curAnimStep;
        for (;;)
        {
            curAnimStep = getCursorAnimationStep();
            if (curAnimStep != prevAnimStep)
                break;
            char key = (char) inkey();
            if (key != 0)
            {
                if (colorsInverted)
                    *cursorPos ^= 0x40;  // restore original colors
                return key;
            }
        }
        prevAnimStep = curAnimStep;
    }
}


void
getCursorPos(byte *col, byte *row)
{
    word offset = * (word *) 0x0088 - VIDRAM;
    *col = (byte) offset & 31;
    *row = (byte) (offset >> 5);
}


void
voidLocate(byte column, byte row)
{
    (void) locate(column, row);
}


void
putcharNoAdvance(int c)
{
    word origCursorPos = * (word *) 0x0088;
    putchar(c);
    * (word *) 0x0088 = origCursorPos;
}


void
printAt(byte col, byte row, const char *line)
{
    locate(col, row);
    putstr(line, strlen(line));
}


void
printRepeatedChar(byte n, char ch)
{
    while (n--)
        putchar(ch);
}


void
clearLine(byte startCol, byte row, byte numChars)
{
    locate(startCol, row);
    printRepeatedChar(numChars, ' ');
    locate(startCol, row);
}


void
scrollSubWindowUp(byte leftCol, byte topRow, byte width, byte height)
{
    byte *dest = (byte *) VIDRAM + ((word) topRow << 5) + leftCol;  // top left of subwindow
    for (byte i = 1; i < height; ++i, dest += 32)  // for each row to copy
        memcpy(dest, dest + 32, width);  // copy a row to the one above it
    memset(dest, 0x60, width);  // clear bottom row of subwindow
}


void
scrollSubWindowDown(byte leftCol, byte topRow, byte width, byte height)
{
    byte *dest = (byte *) VIDRAM + ((word) (topRow + height - 1) << 5) + leftCol;  // left of last row of subwindow
    for (byte i = 1; i < height; ++i, dest -= 32)  // for each row to copy
        memcpy(dest, dest - 32, width);  // copy a row to the one below it
    memset(dest, 0x60, width);  // clear top row of subwindow
}


FileChoiceDialog_Terminal sg4Terminal =
{
    clearLine,
    termHeight,
    highlightLine,
    waitKeyWithAnimatedCursor,
    getCursorPos,
    voidLocate,
    putcharNoAdvance,
    printAt,
    printf,
    putstr,
    printRepeatedChar,
    scrollSubWindowDown,
    scrollSubWindowUp,
};


#endif  /* SG4 (32x16) terminal. */


FileNameCell filenameCellPool[MAX_FILES_PER_DISK];


void
initFilenameCellPool(void)
{
    for (byte i = 0; i < MAX_FILES_PER_DISK; ++i)
    {
        filenameCellPool[i].nameExt[0] = '\0';
        filenameCellPool[i].next = 0xFFFF;  // means free (invalid RAM address on 6809)
    }
}


// Only select the .DAT files.
//
BOOL
filterFileName(const char *name, const char *extension)
{
    return memcmp(extension, "DAT", 3) == 0;
}


void *
allocateFileNameCell(void)
{
    for (byte i = 0; i < MAX_FILES_PER_DISK; ++i)
        if (filenameCellPool[i].next == 0xFFFF)
        {
            filenameCellPool[i].next = NULL;
            return &filenameCellPool[i];
        }
    return NULL;
}


void
freeFileNameCell(FileNameCell *cell)
{
    filenameCellPool[cell - filenameCellPool].next = 0xFFFF;  // now free
}


int
main()
{
    width(32);
    rgb();
    cls(MAIN_SCREEN_BG_COLOR);

    // Initialize the decbfile file access library.
    // Uses Disk Basic's sector access routine.
    //
    DECBDrive driveArray[4];
    byte err = decb_init(driveArray, 4);
    if (err != DECB_OK)
    {
        printf("FAILED TO INIT DECBFILE: ERROR #%u\n", err);
        return 1;
    }
    decb_registerDrive(0);
    decb_registerDrive(1);
    decb_registerDrive(2);
    decb_registerDrive(3);

    initFilenameCellPool();

    byte driveNo = 0;
    for (;;)
    {
        char filename[FileChoiceDialog_maxPathLen + 1];
        if (!FileChoiceDialog_chooseFilePath(&driveNo,
                                            filename,
                                            &sg4Terminal,
                                            filterFileName,
                                            allocateFileNameCell,
                                            freeFileNameCell))
        {
            cls(255);   
            printf("CANCELLED.\n");
        }
        else
        {
            cls(255);
            printf("CHOSEN FILE: %s:%u\n", filename, driveNo);
            
            word numSectors;
            dword lengthInBytes;
            byte numGranules = decb_getFileSizeFromFilename(driveNo, filename, &numSectors, &lengthInBytes);
            if (numGranules == 0xFF)
            {
                printf("FAILED TO GET THE SIZE OF THIS\n"
                       "FILE: ERROR #%u.\n", err);
            }
            else
            {
                printf("SIZE IN BYTES:    %lu\n"
                       "SIZE IN SECTORS:  %u\n"
                       "SIZE IN GRANULES: %u\n",
                                    lengthInBytes, numSectors, numGranules);
            }
        }
        printf("\n"
               "PRESS [ENTER] TO TRY AGAIN\n"
               "OR [BREAK] TO QUIT: ");
        char key = waitKeyWithAnimatedCursor();
        if (key == 3)  // break key
            break;
        cls(MAIN_SCREEN_BG_COLOR);
    }

    decb_shutdown();
    cls(255);
    printf("THIS WAS FCDEMO.BIN (DECBFILE).\n");
    return 0;
}
