/*  loadbin.c - Demo program that loads a .bin file

    By Pierre Sarrazin <http://sarrazip.com/>
    This file is in the public domain.
*/

#include "binfile.h"

#include "decbfile.h"


void
fail(byte errorCode, const char *errorMessage)
{
    printf("ERROR #%u: %s\n", errorCode, errorMessage);
    exit(1);
}


typedef struct FileReader
{
    DECBRecordFile recordFile;
    word sectorIndex;
    byte errorCode;
} FileReader;


// This function gets called by bin_loadBinFile() each time it wants
// to get the next sector of the .bin file.
//
BOOL
sectorLoadFunction(byte secBuf[256], void *userData)
{
    FileReader *pFileReader = (FileReader *) userData;
    byte err = decb_readRecord(&pFileReader->recordFile, secBuf, pFileReader->sectorIndex);
    if (err != DECB_OK)
    {
        pFileReader->errorCode = err;  // remember the error code
        return FALSE;  // tell bin_loadBinFile() to stop
    }
    return TRUE;  // success
}


void
headerCallback(bin_Header *header, void *userData)
{
    if (header->code == 0)
        printf("BLOCK OF %u BYTES AT %p\n", header->n0, header->n1);
    else
        printf("ENTRY POINT AT %p\n", header->n1);
}


void *
loadBinFile(byte driveNo, const char *binFilename)
{
    // Open a .bin file and look at it as a sequence of 256-byte records,
    // because a DECB sector is 256 bytes.
    FileReader fileReader;
    fileReader.sectorIndex = 0;
    byte err = decb_openRecordFile(&fileReader.recordFile, driveNo, binFilename, 0, 256);
    if (err != DECB_OK)
        fail(err, "DECB_OPENRECORDFILE");

    byte sectorBuffer[256];
    void *entryPoint;
    if (!bin_loadBinFile(sectorLoadFunction,
                         headerCallback,
                         sectorBuffer,
                         &fileReader,
                         &entryPoint))
        fail(fileReader.errorCode, "BIN_LOADBINFILE FAILED");

    err = decb_closeRecordFile(&fileReader.recordFile);
    if (err != DECB_OK)
        fail(err, "DECB_CLOSERECORDFILE");

    return entryPoint;
}


int
main()
{
    // Initialize the file access library.
    DECBDrive driveArray[1];
    byte err = decb_init(&driveArray, 1);  // only support drive 0 in this demo
    if (err != DECB_OK)
        fail(err, "DECB_INIT");

    err = decb_registerDrive(0);
    if (err != DECB_OK)
        fail(err, "DECB_REGISTERDRIVE");

    void *entryPoint = loadBinFile(0, "EXAMPLE.BIN");
    
    // Look at the entry point as a pointer to a function that returns an int.
    int (*funcPtr)(void) = (int (*)(void)) entryPoint;

    // Call the routine at the entry point. That routine leaves its return value in D,
    // in compliance with the CMOC calling convention.
    printf("CALLING %p\n", funcPtr);
    int result = (*funcPtr)();
    printf("RETURN VALUE: %u: %s\n", result, result == 6809 ? "SUCCESS" : "FAILURE");

    err = decb_shutdown();
    if (err != DECB_OK)
        fail(err, "DECB_SHUTDOWN");

    return 0;
}
