; rdwrdemo.asm - Demo of an assembly language program that accesses a DECB file.
;
; This file is in the public domain.
;
; This demo requires the Basic interpreter to be present, but the driver
; that it uses does not.
;
; The following symbols must be pre-defined by the assembler:
; - demoStart: The start of this demo program.
; - driver: Address where the driver has already been loaded.
; For example, pass -D demoStart=0x5000 -D driver=0x6000 to lwasm.

        org     demoStart

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

vidram          equ     $0400
vidramEnd       equ     vidram+$0200

curpos          equ     $0088

polcat          equ     $A000
chrout          equ     $A002

twoBlanks       equ     $6060           ; two 6847 codes

cr              equ     13              ; carriage return

; Driver routine offsets.
readFileToMemory        equ     0
writeMemoryToFile       equ     2

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

start:
        leax    s_loadingFile,pcr
        lbsr    printString

        leax    fileToLoad,pcr
        lbsr    printString

        leax    s_toTheScreen,pcr
        lbsr    printString

        jsr     driver                  ; initialize driver, get driver routine table address in D
        tfr     d,u                     ; point to routine table with U

; Stack arguments to reading routine, then call it.
        ldd     #2                      ; number of sectors to load
        pshs    d
        ldd     #vidram                 ; address where to load file contents
        pshs    d
        leax    fileToLoad,pcr          ; point to ASCIIZ of filename to load
        pshs    x
        ldd     #0                      ; drive number of file to load
        pshs    d                       ; byte arguments must be passed as words

        jsr     [readFileToMemory,u]    ; call driver's reading routine to read file into destAddress

        leas    4*2,s                   ; discard 4 routine arguments 

; Routine result code is now in D.
        cmpd    #0                      ; success if 0
        beq     @successfulRead
;
        pshs    d                       ; preserve error code
        leax    s_failedWithErrorCode,pcr
        lbsr    printString
;
        puls    d                       ; restore error code
        lbsr    printWordInHex          ; print it
        bra     exit
;
@successfulRead:
        lbsr    waitForKey
;
; Prepare a 512-byte buffer on the stack to be written to a disk file.
;
        leas    -512,s
        
        leax    ,s                      ; point X to buffer
        lda     #'A'
        ldb     #512/4                  ; number of 4-byte regions in buffer
@spaceFillLoop
        sta     ,x+                     ; write a letter 3 times
        sta     ,x+
        sta     ,x+
        pshs    a
        lda     #cr                     ; write a carriage return
        sta     ,x+
        puls    a
        inca                            ; use next letter during next iteration
        cmpa    #'Z'
        bls     @noWrap
        lda     #'A'
@noWrap
        decb                            ; if not finished filling buffer
        bne     @spaceFillLoop

; Write buffer at S to a file.

        bsr     clrscr

        leax    s_fileGettingWritten,pcr
        bsr     printString

        leax    ,s                      ; keep address of contents to write to file

        ldd     #1                      ; 1=text, 0=binary
        pshs    d                       ; byte arguments must be passed as words
        ldd     #2                      ; number of sectors to write
        pshs    d
        pshs    x                       ; push address of contents to write to file
        leax    fileToSave,pcr          ; point to ASCIIZ of filename to create
        pshs    x
        ldd     #0                      ; drive number of file to create
        pshs    d                       ; byte arguments must be passed as words

        jsr     [writeMemoryToFile,u]   ; call driver's reading routine to read file into destAddress

        leas    5*2,s                   ; discard 5 routine arguments 

        leas    512,s                   ; discard buffer just written

        leax    s_done,pcr
        bsr     printString

exit:
        lda     #cr
        jsr     [chrout]
        rts

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

; Writes at vidram. Homes the cursor.
; Trashes D, X.
;
clrscr:
        ldx     #vidram
        ldd     #twoBlanks
@clear
        std     ,x++
        cmpx    #vidramEnd
        blo     @clear

        ldd     #vidram         ; home the cursor
        std     curpos
        rts

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

; Input: A = column (0..31), B = row (0..15).
; Trashes D, X.
;
locate
        pshs    a
        lda     #32
        mul
        tfr     d,x     ; 0..480
        puls    b       ; column
        abx
        leax    vidram,x
        stx     curpos
        rts

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

; printString: Prints ASCIIZ at X via chrout.
; May trash all registers.
;
@loop
        jsr     [chrout]
printString:
        lda     ,x+
        bne     @loop
        rts

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

; Prints D in 4 hex digits via chrout.
; May trash all registers.
;
printWordInHex:
        pshs    b                       ; preserve low byte
        bsr     printByteInHex          ; print high byte (A)
        puls    a                       ; restore low byte

        ; FALLTHROUGH to print low byte

; Prints A in 2 hex digits via chrout.
; May trash all registers.
;
printByteInHex:
        pshs    a
        lsra
        lsra
        lsra
        lsra                    ; A is now high nybble of byte to print
        leax    hexDigits,pcr
        lda     a,x
        jsr     [chrout]

        puls    a               ; restore original byte
        anda    #$0F            ; A is now low nybble of byte to print
        leax    hexDigits,pcr
        lda     a,x
        jmp     [chrout]

hexDigits:
        fcc     "0123456789ABCDEF"

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

waitForKey:
        jsr     [polcat]
        beq     waitForKey
        rts

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

fileToLoad:
        fcc     "INPUT.TXT"
        fcb     0
fileToSave:
        fcc     "OUTPUT.TXT"
        fcb     0

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

s_loadingFile:
        fcc     "LOADING FILE "
        fcb     0
s_toTheScreen:
        fcb     13
        fcc     "TO THE SCREEN..."
        fcb     13,0
s_success:
        fcc     "SUCCESS."
        fcb     0
s_failedWithErrorCode:
        fcc     "FAILED WITH ERROR $"
        fcb     0
s_fileGettingWritten
        fcc     "DATA IS GETTING WRITTEN"
        fcb     13
        fcc     "TO A FILE. PLEASE WAIT..."
        fcb     13
        fcb     0
;s_pleaseWait
;        fcc     "PLEASE WAIT... "
;        fcb     0
s_done
        fcc     "DONE."
        fcb     13
        fcb     0

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

        end     start
