/*  decbfile - A read/write DECB file library.

    By Pierre Sarrazin <http://sarrazip.com/>
    This file is in the public domain.
*/

#include "decbfile_private.h"


// Private function.
//
// Changes the length of a currently open file.
// The new length must not require the allocation of new granules to the file.
//
// numGranulesWithSuccessor: The new number of granules that are followed by
//                           another granule in the file.
//                           May be zero.
//                           Must not equal or exceed DECB_MAX_NUM_GRANULES.
// numSectorsInLastGranule: The new number of used sectors in the file's
//                          new last granule (1..9).
// numBytesUsedInLastSector: The new number of bytes used by the file's
//                           last sector (0..256).
//
// Returns DECB_ERR_INVALID_ARGUMENT, or an error code if the FAT could
// not be read, or DECB_ERR_CORRUPT_FAT, or DECB_OK upon success.
// DECB_OK is returned when the specified length is longer than the current file.
//
// This function can be used to lengthen the file, but not to the point of
// requiring the allocation of new granules.
//
// Upon success, decb_closeSectorFile() must be called for the truncation
// to take effect on the physical disk.
// There is no need to call decb_setNumBytesUsedInLastSector().
//
byte
decb_truncateOpenFileInGranules(DECBFile *file,
                                byte numGranulesWithSuccessor,
                                byte numSectorsInLastGranule,
                                word numBytesUsedInLastSector)
{
    if (numSectorsInLastGranule == 0 || numSectorsInLastGranule > 9
            || numBytesUsedInLastSector > 256)
        return DECB_ERR_INVALID_ARGUMENT;
    if (numGranulesWithSuccessor >= DECB_MAX_NUM_GRANULES)
        return DECB_OK;
    byte driveNo = file->driveNo;
    byte err = decb_readFAT(driveNo);
    if (err != DECB_OK)
        return err;
    DECBDrive *drv = decb_getDrive(driveNo);
    if (!drv)
        return DECB_ERR_DRIVE_NOT_REGISTERED;
    byte *fat = drv->fatBuffer;
    byte granIndex = 0;
    byte wroteLastEntry = FALSE;
    byte g = file->firstGranule;
    while (g < 0xC0)  // while 'g' points to a successor FAT entry index
    {
        byte next = fat[g];
        if (granIndex == numGranulesWithSuccessor)  // if g is new last entry
        {
            err = decb_setFATEntry(driveNo, g, 0xC0 | numSectorsInLastGranule);
            if (err != DECB_OK)
                return err;
            wroteLastEntry = TRUE;
        }
        else if (granIndex > numGranulesWithSuccessor)  // if g is past new last entry
        {
            err = decb_setFATEntry(driveNo, g, 0xFF);
            if (err != DECB_OK)
                return err;
        }
        ++granIndex;
        g = next;
        if (!decb_isValidFATEntry(g))
            return DECB_ERR_CORRUPT_FAT;
    }
    if (!wroteLastEntry)
        return DECB_OK;
    file->modified = TRUE;
    decb_setNumBytesUsedInLastSector(file, numBytesUsedInLastSector);
    return DECB_OK;
}
