/*  decbfile - A read/write DECB file library.

    By Pierre Sarrazin <http://sarrazip.com/>
    This file is in the public domain.
*/

#include "decbfile_private.h"


byte
decb_writeHeader(DECBRecordFile *recFile, byte *header)
{
    if (recFile->headerSize == 0)
        return DECB_OK;
    // The header may span more than one sector.
    // The last of these sectors may only be partially occupied by the header.
    byte numHeaderSectors = (byte) ((recFile->headerSize + 255) >> 8);
    byte lastHeaderSectorIndex = numHeaderSectors - 1;
    for (byte sectorIndex = 0; sectorIndex < lastHeaderSectorIndex; ++sectorIndex)  // for each complete sector
    {
        byte err = decb_writeSector(&recFile->file, header + (((word) sectorIndex) << 8), sectorIndex);
        if (err != DECB_OK)
            return err;
    }
    // Read the header's last sector.
    // Change only part of it (or all of it, if the header size is divisible by the sector size).
    // Write the sector back.
    byte sectorBuffer[256];
    byte err = decb_readSector(&recFile->file, sectorBuffer, lastHeaderSectorIndex);
    if (err == DECB_ERR_NOT_FOUND)
    {
        // The file is not long enough yet to have that sector.
        // Fill the sector buffer (to avoid writing random memory contents to disk), then continue.
        // The call to decb_writeSector() will extend the file up to the written sector.
        memset(sectorBuffer, 0xFF, 256);
    }
    else if (err != DECB_OK)
        return err;
    word memcpySize = recFile->headerSize & 0x00FF;
    if (memcpySize == 0)
        memcpySize = 256;
    memcpy(sectorBuffer, header + (((word) lastHeaderSectorIndex) << 8), memcpySize);
    err = decb_writeSector(&recFile->file, sectorBuffer, lastHeaderSectorIndex);
    if (err != DECB_OK)
        return err;
    // If the file currently has no records, then the header is the EOF.
    // Update numBytesUsedInLastSector in such a case.
    if (recFile->numRecords == 0)
        recFile->file.numBytesUsedInLastSector = memcpySize;
    return DECB_OK;
}
