/*  decbfile - A read/write DECB file library.

    By Pierre Sarrazin <http://sarrazip.com/>
    This file is in the public domain.
*/

#include "decbfile_private.h"


byte
decb_writeRecord(DECBRecordFile *recFile, byte *record, word recordIndex)
{
    word sectorIndex;
    byte byteOffsetInSector;
    byte err = decb_getRecordOffsets(recFile, recordIndex, &sectorIndex, &byteOffsetInSector);
    if (err != DECB_OK)
        return err;
    word recordSizeToWrite = recFile->recordSize;
    for (;;)
    {
        // Read a sector that contains the record, or part of the record.
        byte sectorBuffer[256];
        err = decb_readSector(&recFile->file, sectorBuffer, sectorIndex);
        if (err == DECB_ERR_NOT_FOUND)
        {
            // The file is not long enough yet to have the sector that contains the record to write.
            // Fill the sector buffer (to avoid writing random memory contents to disk), then continue.
            // The call to decb_writeSector() will extend the file up to the written sector.
            memset(sectorBuffer, 0xFF, 256);
        }
        else if (err != DECB_OK)
            return err;
        // Copy the record, or part of the record, to the sector buffer.
        word roomInCurrentSector = 256 - byteOffsetInSector;
        byte isLastSectorOfRecord = (recordSizeToWrite <= roomInCurrentSector);
        word memcpySize = (isLastSectorOfRecord ? recordSizeToWrite : roomInCurrentSector);
        memcpy(sectorBuffer + byteOffsetInSector, record, memcpySize);
        // Write the sector back to disk.
        err = decb_writeSector(&recFile->file, sectorBuffer, sectorIndex);
        if (err != DECB_OK)
            return err;
        if (isLastSectorOfRecord)
        {
            // If the file has just been extended, update the number of records.
            if (recordIndex + 1 > recFile->numRecords)
                recFile->numRecords = recordIndex + 1;
            // If the written record is the last one in the file, update
            // the length of the file's last sector.
            //
            if (recordIndex == recFile->numRecords - 1)
                recFile->file.numBytesUsedInLastSector = byteOffsetInSector + memcpySize;
            break;
        }
        byteOffsetInSector = 0;  // in next sector, start writing at offset 0
        recordSizeToWrite -= memcpySize;  // this much less is left to be written
        record += memcpySize;  // point to next part to be written
        ++sectorIndex;  // next sector in file
    }
    return DECB_OK;
}
